import os
import logging

import yp.client
from infra.yp_drp import config, exclusiveservice, http
from infra.yp_drp import yp_client, yp_poller
from infra.skylib.debug_server.reverse import ReplServer


log = logging.getLogger('drp')


class Application:
    def __init__(self, cfg):
        self.cfg = cfg

        repl_path = config.get_value(cfg, 'debug.sock_path')
        if repl_path:
            self.repl_server = ReplServer(sock_path=repl_path)
            self.repl_server.start()
        else:
            self.repl_server = None

        port = int(config.get_value(cfg, 'http.port', 80))
        self.statistics = http.StatisticsServer(port=port)

        token = config.get_value(cfg, 'yp.token', os.getenv('YP_TOKEN'))
        max_receive_message_length = int(config.get_value(cfg, 'yp.max_receive_message_length', 32 * (2 ** 20)))
        yp_client_base = yp.client.YpClient(
            address=config.get_value(cfg, 'yp.address'),
            config={
                'token': token,
                'body_log_size_limit': 1,
                'grpc_channel_options': {
                    'max_receive_message_length': max_receive_message_length,
                },
            },
        )
        yp_client_stub = yp_client_base.create_grpc_object_stub()
        yp_drp_client = yp_client.YpClient(yp_client_stub)
        self.yp_poller = yp_poller.YpPoller(
            yp_client=yp_drp_client,
            batch_size=int(config.get_value(cfg, 'yp.poll.batch_size')),
            username=config.get_value(cfg, 'yp.username'),
            statistics=self.statistics,
        )

        coord_cfg = config.get_value(cfg, 'coord')
        self.poller = exclusiveservice.ExclusiveService(
            coord_cfg,
            f'drp-{coord_cfg["proxy"]}',
            self.yp_poller,
        )

    # Public methods

    async def run(self):
        """
        Start application.
        Blocks until stop was called.
        """
        log.info('starting service...')
        await self.statistics.start()
        await self.poller.run()

    async def stop(self):
        """
        Gracefully stop application.
        Can block for a long time or throw exception, be ready.
        """
        log.info('stopping service...')
        await self.poller.stop()
        await self.statistics.shutdown()
        log.info('=' * 30)
        if self.repl_server:
            self.repl_server.stop_repl()
