import logging
import itertools
from collections import defaultdict
from typing import List, MutableMapping, Optional, Any, Dict, Tuple

from infra.yp_drp import podutil, yp_client
from infra.yp_drp.podutil import Resource


log = logging.getLogger('updater')


class DynamicResourceUpdater:
    def __init__(self, client: yp_client.YpClient) -> None:
        self.client = client
        self.scheduled_resource_removals = []
        self.scheduled_resource_status_updates = []
        self.scheduled_pod_status_updates = []
        self.scheduled_pod_spec_updates = []

    async def commit(self, batch_size: int) -> None:
        log = logging.getLogger('updater')
        log.debug(f"commiting {len(self.scheduled_resource_removals)} resource removals")

        await self.client.remove_resources(self.scheduled_resource_removals, batch_size)
        log.debug(f"commited {len(self.scheduled_resource_removals)} resource removals")
        self.scheduled_resource_removals = []

        log.debug(f"commiting {len(self.scheduled_resource_status_updates)} resource status updates")
        await self.client.update_resource_statuses(self.scheduled_resource_status_updates, batch_size)
        log.debug(f"commited {len(self.scheduled_resource_status_updates)} resource status updates")
        self.scheduled_resource_status_updates = []

        log.debug(f"commiting {len(self.scheduled_pod_status_updates)} pod status updates")
        await self.client.update_pod_resource_statuses(self.scheduled_pod_status_updates, batch_size)
        log.debug(f"commited {len(self.scheduled_pod_status_updates)} pod status updates")
        self.scheduled_pod_status_updates = []

        # remove duplicate pods
        self.scheduled_pod_spec_updates = list(map(next, map(lambda x: x[1], itertools.groupby(
            sorted(self.scheduled_pod_spec_updates, key=lambda pod: pod.meta.id),
            key=lambda pod: pod.meta.id
        ))))
        log.debug(f"commiting {len(self.scheduled_pod_spec_updates)} pod spec updates")
        await self.client.update_pod_resource_specs(self.scheduled_pod_spec_updates, batch_size)
        log.debug(f"commited {len(self.scheduled_pod_spec_updates)} pod spec updates")
        self.scheduled_pod_spec_updates = []

    async def sync_pod_status(self, pod: Resource.Pod) -> None:
        # TODO when there will be an pod-interface in container, this step will be redundant
        resources = podutil.resources_to_dict(pod.spec_dynamic_resources,
                                              pod.status_dynamic_resources)
        workloads = pod.status_workloads
        box_by_dru = podutil.find_dru_workloads(pod)
        status_changed = False

        for workload in filter(lambda w: w.get('id') in box_by_dru, workloads):
            box_id = box_by_dru[workload.get('id')]
            status_changed |= podutil.update_resource_statuses(
                resources[box_id],
                workload.get('start', {}).get('current', {}).get('stdout_tail')
            )

        statuses = [value
                    for box in resources.values()
                    for value in box.values()
                    if value is not None
                    ]

        status_changed |= (len(statuses) != len(pod.status_dynamic_resources))
        del pod.status_dynamic_resources[:]
        pod.status_dynamic_resources.extend(statuses)

        if status_changed:
            self.scheduled_pod_status_updates.append(pod)

    async def count_resource_statuses(
        self,
        pod: Resource.Pod,
        new_statuses: MutableMapping[str, Any],
    ) -> None:
        for dynresource in pod.status_dynamic_resources:
            resource_id = dynresource.get('id')
            status = new_statuses.setdefault(resource_id, defaultdict(lambda: defaultdict(int)))

            for condition in ('ready', 'in_progress', 'error'):
                if (dynresource or {}).get(condition, {}).get('status') == 'true':
                    status[dynresource.get('revision')][condition] += 1

    async def sync_resource_status(
        self,
        resource_id: str,
        resource_status: dict,
        statuses: MutableMapping[str, MutableMapping[str, int]],
        update_window: int,
        pods_total: int,
    ) -> None:
        status_changed = False
        old_status = {}
        for rev in resource_status.get('revisions', []):
            for condition in ("ready", "in_progress", "error"):
                old_status[(rev.get('revision', 0), condition)] = (rev or {}).get(condition, {}).get('pod_count', 0)

        resource_status['revisions'] = []

        done_threshold = max(1, int(pods_total * 0.9))  # TODO take from spec

        for revision, states in statuses.items():
            state = {}
            state['revision'] = revision
            for condition, count in states.items():
                state.setdefault(condition, {})['pod_count'] = count
                if old_status.get((revision, condition), 0) != count:
                    status_changed = True
                old_status.pop((revision, condition), None)

            is_ready = state.get('ready', {}).get('pod_count', 0) >= done_threshold
            state.setdefault('ready', {}).setdefault('condition', {})['status'] = podutil.yp_bool(is_ready)
            state.setdefault('in_progress', {}).setdefault('condition', {})['status'] = podutil.yp_bool(
                state.get('in_progress', {}).get('pod_count', 0) > 0 and not is_ready
            )
            state.setdefault('error', {}).setdefault('condition', {})['status'] = podutil.yp_bool(
                state.get('error', {}).get('pod_count', 0) > 0
            )
            resource_status['revisions'].append(state)

        if any(old_status.values()) or status_changed:  # we intentionally skip zero-values
            self.scheduled_resource_status_updates.append((resource_id, resource_status))

    async def update_allocations(
        self,
        resource_id: str,
        resource: dict,
        pods: List[Resource.Pod],
        expired: bool,
    ) -> None:
        if expired:
            log.info(f'[{resource_id}] removing resource')
            self.scheduled_resource_removals.append(resource_id)
            return

        # log.debug(f"[{resource_id}] updating specs")

        window = resource.get('update_window', 0)
        ready = in_progress = 0
        total = len(pods)
        for pod in pods:
            rspec = find_resource(pod.spec_dynamic_resources, resource_id)
            rstatus = find_resource(pod.status_dynamic_resources, resource_id)
            if rspec and resource.get('revision') == rspec.get('revision'):
                if (
                    not rstatus
                    or rstatus.get('revision') != rspec.get('revision')
                    or rstatus.get('ready', {}).get('status') != 'true'
                ):
                    in_progress += 1
                else:
                    ready += 1

        # log.debug(f"[{resource_id}] found total={total} ready={ready} in_progress={in_progress} window={window}")
        if in_progress >= window:
            return

        to_update = max(0, min(total - ready - in_progress, window - in_progress))
        updated_pods = []

        pods_with_specs = sorted(
            map(lambda pod: get_matching_spec_id(pod, resource), pods),
            key=lambda item: item[0],
        )

        # log.debug(f"[{resource_id}] will schedule {to_update} pods from {len(pods)}")
        for idx, spec, pod in pods_with_specs:
            if not to_update:
                break

            if spec is None:
                log.info(f"[{resource_id}] pod {pod.meta.id} skipped, exactly one matching deploy group required")
                continue

            rspec = find_resource(pod.spec_dynamic_resources, resource_id)
            rstatus = find_resource(pod.status_dynamic_resources, resource_id)
            if rspec and resource.get('revision') == rspec.get('revision'):
                continue

            to_update -= 1
            if not rspec:
                rspec = {'id': resource_id}
                pod.spec_dynamic_resources.append(rspec)
            rspec['revision'] = resource.get('revision')
            # rspec.mark = spec.mark
            rspec['urls'] = list(spec.get('urls', []))
            rspec['storage_options'] = spec.get('storage_options', {}).copy()
            updated_pods.append(pod)

        self.scheduled_pod_spec_updates.extend(updated_pods)
        # log.debug(f"[{resource_id}] scheduled additional {len(updated_pods)} pods")

    async def drop_old_resources(
        self,
        pods: List[Any],
        resources: Dict[str, Any],
        batch_size: int,
    ) -> None:
        for pod in pods:
            drs = [spec for spec in pod.spec_dynamic_resources if spec.get('id') in resources]
            del pod.spec_dynamic_resources[:]
            pod.spec_dynamic_resources.extend(drs)
        await self.client.update_pod_resource_specs(pods, batch_size)


def find_resource(resources: Optional[List[Any]], r_id: str) -> Optional[Any]:
    if resources is not None:
        for r in resources:
            if r.get('id') == r_id:
                return r
    return None


def get_matching_spec_id(pod: Resource.Pod, spec: dict) -> Tuple[int, Optional[Any], Resource.Pod]:
    result = -1, None, pod
    for idx, group in enumerate(spec.get('deploy_groups', [])):
        if podutil.check_labels_matched(pod.labels, group.get('required_labels', {})):
            if result[1] is not None:
                return -1, None, pod
            result = idx, group, pod
    return result
