import simplejson as json
import functools
import itertools
import collections
from typing import Iterable, Any, List, Dict, Optional

import yt_yson_bindings


class Resource:
    __slots__ = ['resource', 'expired', 'pods', 'managed']

    class DynResource:
        __slots__ = ['meta', 'spec', 'status', 'deploy_engine']

        def __init__(self, meta=None, spec=None, status=None, deploy_engine=None):
            self.meta = meta
            self.spec = spec
            self.status = status
            self.deploy_engine = deploy_engine

    class Pod:
        __slots__ = ['meta', 'spec_dynamic_resources', 'spec_workloads', 'labels', 'status_dynamic_resources', 'status_workloads']

        def __init__(self, meta=None, spec_dynamic_resources=None, spec_workloads=None, labels=None, status_dynamic_resources=None, status_workloads=None):
            self.meta = meta
            self.spec_dynamic_resources = spec_dynamic_resources
            self.spec_workloads = spec_workloads
            self.labels = labels
            self.status_dynamic_resources = status_dynamic_resources
            self.status_workloads = status_workloads

    resource: DynResource
    expired: bool
    pods: List[Pod]
    managed: bool

    def __init__(self):
        self.resource = None
        self.expired = False
        self.pods = []
        self.managed = True


def yson_to_proto(yson_response, proto_type):
    return yt_yson_bindings.loads_proto(yson_response.yson, proto_type, skip_unknown_fields=True)


def yson_to_proto_list(yson_response, proto_type):
    raw = yt_yson_bindings.loads(yson_response.yson)
    if not raw or not isinstance(raw, list):
        return []

    return [
        yt_yson_bindings.loads_proto(yt_yson_bindings.dumps(item), proto_type, skip_unknown_fields=True)
        for item in raw
    ]


def resources_to_dict(
    specs: List[dict],
    statuses: List[dict],
) -> Dict[str, Dict[str, dict]]:
    result = collections.defaultdict(dict)
    box_by_res = {}
    for res in specs:
        box_ref = box_by_res[res.get('id')] = res.get('storage_options', {}).get('box_ref')
        result[box_ref][res.get('id')] = None  # make a stub for resources not known to DRU yet
    for res in statuses:
        box_id = box_by_res.get(res.get('id'))  # box can be empty if resource is already removed
        result[box_id][res.get('id')] = res
    return result


def find_dru_workloads(pod: Resource.Pod) -> Dict[str, str]:
    result = {}
    for workload in pod.spec_workloads:
        if workload.get('id', '').endswith("__dru"):  # FIXME can we use some label here?
            result[workload['id']] = workload.get('box_ref')
    return result


def yp_bool(b):
    return 'true' if b else 'false'


def update_resource_statuses(
    box: Dict[str, dict],
    stdout: Optional[bytes],
) -> bool:
    if not stdout:
        return False

    decoder = json.JSONDecoder()
    result = False

    def update_resource(status):
        nonlocal box
        nonlocal result
        resource_id = status['resource_id']
        ready = status['ready']
        in_progress = status['in_progress']
        error = status['error']
        revision = status['revision']
        reason = status.get('reason') or ''

        state = box.get(resource_id)
        if state is None:
            state = box[resource_id] = {}
            result |= True

        result |= (
            state.get('revision', 0) != revision
            or state.get('ready', {}).get('status') != yp_bool(ready)
            or state.get('in_progress', {}).get('status') != yp_bool(in_progress)
            or state.get('error', {}).get('status') != yp_bool(error)
            or state.get('error', {}).get('reason', '') != reason
        )
        state['id'] = resource_id
        state['revision'] = revision
        state.setdefault('ready', {})['status'] = yp_bool(ready)
        state.setdefault('in_progress', {})['status'] = yp_bool(in_progress)
        state.setdefault('error', {})['status'] = yp_bool(error)
        state.setdefault('error', {})['reason'] = reason

    pos = stdout.find('{')  # the output can be truncated, so we find the record start
    while pos != -1:
        try:
            event, endpos = decoder.raw_decode(stdout[pos:])
            kind = event.get('kind')
            if kind == 'event':
                update_resource(event)
            elif kind == 'state':
                for state in event['states']:
                    update_resource(state)

        except Exception:
            pos = stdout.find('{', pos + 1)
        else:
            pos = stdout.find('{', pos + endpos)

    return result


def check_labels_matched(labels: Dict[str, Any], match_labels: Dict[str, Any]) -> bool:
    for k, v1 in match_labels.items():
        v0 = labels.get(k)
        if v0 != v1:
            return False

    return True


def chunk(chunk_size: int, iterable: Iterable[Any]) -> List[Any]:
    return list(itertools.islice(iterable, chunk_size))


def window(iterable: Iterable[Any], chunk_size: int) -> Iterable[List[Any]]:
    return iter(functools.partial(chunk, chunk_size, iter(iterable)), [])
