import logging
import asyncio
from typing import Any, AsyncIterator, Optional, List, Tuple

import yt_yson_bindings
from yp import data_model
from yp.common import GrpcResourceExhaustedError
from yp_proto.yp.client.api.proto import object_service_pb2

from .podutil import Resource, yson_to_proto


class YpClient:
    class NoAcl(Exception):
        pass

    def __init__(self, slave):
        self.slave = slave
        self.loop = asyncio.get_event_loop()
        self.log = logging.getLogger('yp-client')
        # self.loop.set_default_executor(concurent.futures.ThreadPoolExecutor(max_workers=10))  # FIXME

    async def _apply(self, fn, *args) -> Any:
        return await asyncio.get_event_loop().run_in_executor(None, fn, *args)

    async def _select_objects_values(
        self,
        object_type: int,
        batch_size: int,
        selectors: Optional[List[str]] = None,
        timestamp: Optional[int] = None,
    ) -> AsyncIterator[List[bytes]]:
        req = object_service_pb2.TReqSelectObjects()
        req.object_type = object_type
        if timestamp is not None:
            req.timestamp = timestamp
        selectors = selectors or ['']
        req.selector.paths.extend(selectors)
        req.format = object_service_pb2.PF_YSON
        req.options.limit = batch_size

        total = 0
        while True:
            try:
                resp = await self._apply(self.slave.SelectObjects, req)
            except GrpcResourceExhaustedError:
                if req.options.limit < 2:
                    raise
                req.options.limit //= 2
                continue

            read = 0

            for r in resp.results:
                read += 1
                total += 1
                # assert r.meta.object_type == object_type, f"Server does not support object_type={object_type}"
                yield r.value_payloads

            if read < req.options.limit:
                break
            else:
                req.options.continuation_token = resp.continuation_token

    async def generate_timestamp(self) -> int:
        req = object_service_pb2.TReqGenerateTimestamp()
        resp = await self._apply(self.slave.GenerateTimestamp, req)
        return resp.timestamp

    async def list_pod_sets(
        self,
        batch_size: int,
        timestamp: Optional[int] = None,
    ) -> AsyncIterator[str]:
        pod_sets = self._select_objects_values(
            object_type=data_model.OT_POD_SET,
            batch_size=batch_size,
            selectors=['/meta/id'],
            timestamp=timestamp
        )
        async for p in pod_sets:
            yield yt_yson_bindings.loads(p[0].yson)

    async def select_pods(
        self,
        batch_size: int,
        timestamp: Optional[int] = None,
    ) -> AsyncIterator[Resource.Pod]:
        pods = self._select_objects_values(
            object_type=data_model.OT_POD,
            batch_size=batch_size,
            selectors=[
                '/meta',
                '/spec/pod_agent_payload/spec/workloads',
                '/status/dynamic_resources',
                '/labels',
                '/status/agent/pod_agent_payload/status/workloads',
                '/spec/dynamic_resources'
            ],
            timestamp=timestamp,
        )
        async for p in pods:
            try:
                meta = yson_to_proto(p[0], data_model.TPodMeta)
            except Exception:
                self.log.warning("skipped unparsable unknown pod, meta: %r", p[0])
                continue

            try:
                yield Resource.Pod(
                    meta=meta,
                    spec_workloads=yt_yson_bindings.loads(p[1].yson) or [],
                    spec_dynamic_resources=yt_yson_bindings.loads(p[5].yson) or [],
                    status_dynamic_resources=yt_yson_bindings.loads(p[2].yson) or [],
                    status_workloads=yt_yson_bindings.loads(p[4].yson) or [],
                    labels=yt_yson_bindings.loads(p[3].yson) or {},
                )
            except Exception:
                self.log.warning("skipped unparsable pod %r", meta.id)
                # raise Exception("failed to load pod %s" % meta.id)
                continue

    async def select_resources(
        self,
        batch_size: int,
        timestamp: Optional[int] = None,
    ) -> AsyncIterator[Resource.DynResource]:
        pods = self._select_objects_values(
            object_type=data_model.OT_DYNAMIC_RESOURCE,
            batch_size=batch_size,
            selectors=['/meta', '/spec', '/status', '/labels/deploy_engine'],
            timestamp=timestamp,
        )
        async for p in pods:
            yield Resource.DynResource(
                meta=yson_to_proto(p[0], data_model.TDynamicResourceMeta),
                spec=yt_yson_bindings.loads(p[1].yson),
                status=yt_yson_bindings.loads(p[2].yson),
                deploy_engine=yt_yson_bindings.loads(p[3].yson) or 'drp',
            )

    async def update_pod_resource_statuses(
        self,
        pods: List[Resource.Pod],
        batch_size: int,
    ) -> None:
        offset = 0
        while offset < len(pods):
            req = object_service_pb2.TReqUpdateObjects()
            for pod in pods[offset:offset + batch_size]:
                subreq = req.subrequests.add()
                subreq.object_type = data_model.OT_POD
                subreq.object_id = pod.meta.id
                upd = subreq.set_updates.add()
                upd.path = '/status/dynamic_resources'
                upd.value_payload.yson = yt_yson_bindings.dumps(pod.status_dynamic_resources)

            try:
                await self._apply(self.slave.UpdateObjects, req)
            except GrpcResourceExhaustedError:
                if batch_size < 2:
                    raise
                batch_size //= 2
                continue

            offset += batch_size

    async def update_pod_resource_specs(
        self,
        pods: List[Any],
        batch_size: int,
    ) -> None:
        offset = 0
        while offset < len(pods):
            req = object_service_pb2.TReqUpdateObjects()
            for pod in pods[offset:offset + batch_size]:
                subreq = req.subrequests.add()
                subreq.object_type = data_model.OT_POD
                subreq.object_id = pod.meta.id
                upd = subreq.set_updates.add()
                upd.path = '/spec/dynamic_resources'
                upd.value_payload.yson = yt_yson_bindings.dumps(pod.spec_dynamic_resources)

            try:
                await self._apply(self.slave.UpdateObjects, req)
            except GrpcResourceExhaustedError:
                if batch_size < 2:
                    raise
                batch_size //= 2
                continue

            offset += batch_size

    async def update_resource_statuses(
        self,
        resources: List[Tuple[str, dict]],
        batch_size: int,
    ) -> None:
        offset = 0
        while offset < len(resources):
            req = object_service_pb2.TReqUpdateObjects()
            for resource_id, status in resources[offset:offset + batch_size]:
                subreq = req.subrequests.add()
                subreq.object_type = data_model.OT_DYNAMIC_RESOURCE
                subreq.object_id = resource_id
                upd = subreq.set_updates.add()
                upd.path = '/status'
                upd.value_payload.yson = yt_yson_bindings.dumps(status)

            try:
                await self._apply(self.slave.UpdateObjects, req)
            except GrpcResourceExhaustedError:
                if batch_size < 2:
                    raise
                batch_size //= 2
                continue

            offset += batch_size

    async def remove_resources(
        self,
        resources: List[str],
        batch_size: int,
    ) -> None:
        offset = 0
        while offset < len(resources):
            req = object_service_pb2.TReqRemoveObjects()
            for resource in resources[offset:offset + batch_size]:
                sub = req.subrequests.add()
                sub.object_type = data_model.OT_DYNAMIC_RESOURCE
                sub.object_id = resource

            try:
                await self._apply(self.slave.RemoveObjects, req)
            except GrpcResourceExhaustedError:
                if batch_size < 2:
                    raise
                batch_size //= 2
                continue

            offset += batch_size
