import os
import getpass
import argparse

from yp import data_model
import yp.client as ypc
import yp_proto.yp.client.api.proto.object_service_pb2 as osp
import yt_yson_bindings


def parse_args():
    parser = argparse.ArgumentParser(usage='show or update dynamic resource')
    parser.add_argument('--address', help='yp cluster', default='sas-test')
    modes = parser.add_subparsers(help='modes', metavar='MODE', dest='mode')
    update = modes.add_parser('update', help='update resource spec')
    update.add_argument('resource_id', help='resource uniq id')
    update.add_argument('updates', help='key.subkey=value', nargs='+')

    show = modes.add_parser('show', help='show resource spec')
    show.add_argument('resource_id', help='resource uniq id')

    remove = modes.add_parser('remove', help='remove resource')
    remove.add_argument('resource_id', help='resource uniq id')

    create = modes.add_parser('create', help='create resource')
    create.add_argument('pod_set_id', help='if of pod_set to bind to')

    args = parser.parse_args()
    if not args.mode:
        parser.print_help()
        raise SystemExit(1)

    if '.' not in args.address:
        args.address = f'{args.address}.yp.yandex.net:8090'

    return args


def print_resource(cc, resource_id):
    req = osp.TReqGetObject()
    req.object_type = data_model.OT_DYNAMIC_RESOURCE
    req.object_id = resource_id
    req.selector.paths[:] = ['']
    req.format = osp.PF_PROTOBUF
    rsp = cc.GetObject(req).result
    if not rsp or not rsp.value_payloads:
        print("Not found")
    print(data_model.TDynamicResource.FromString(rsp.value_payloads[0].protobuf))


def main():
    args = parse_args()

    c = ypc.YpClient(address=args.address, config={'token': open(os.path.expanduser('~/.yp/token')).read().strip()})
    cc = c.create_grpc_object_stub()

    if args.mode == 'show':
        resource_id = args.resource_id
        print_resource(cc, resource_id)

    elif args.mode == 'remove':
        resource_id = args.resource_id

        req = osp.TReqRemoveObject()
        req.object_type = data_model.OT_DYNAMIC_RESOURCE
        req.object_id = resource_id

        rsp = cc.RemoveObject(req)
        print(rsp)

    elif args.mode == 'create':
        obj = data_model.TDynamicResource()
        obj.meta.pod_set_id = args.pod_set_id
        acl = obj.meta.acl.add()
        acl.action = data_model.ACA_ALLOW
        acl.permissions.extend([
            data_model.ACP_READ,
            data_model.ACA_WRITE,
            data_model.ACA_CREATE,
        ])
        acl.subjects.append(getpass.getuser())
        acl.subjects.append('robot-yp-dynresource')
        req = osp.TReqCreateObject(
            object_type=data_model.OT_DYNAMIC_RESOURCE,
            attributes=yt_yson_bindings.dumps_proto(obj)
        )
        rsp = cc.CreateObject(req)
        print(rsp)

    elif args.mode == 'update':
        resource_id = args.resource_id

        req = osp.TReqUpdateObject()
        req.object_type = data_model.OT_DYNAMIC_RESOURCE
        req.object_id = resource_id
        for update in args.updates:
            path, value = update.split('=', 1)
            if value == '{}':
                value = {}
            else:
                try:
                    value = int(value)
                except Exception:
                    pass
            path = '/' + path.replace('.', '/')
            subreq = req.set_updates.add()
            subreq.path = path
            subreq.value = yt_yson_bindings.dumps(value)

        rsp = cc.UpdateObject(req)
        print_resource(cc, resource_id)
