import os
import re
import json
import socket
import asyncio
import logging
import logging.handlers
import argparse
from typing import Optional

from infra.yp_dru.updater import process_all, fetch_resources_info, NoChanges
from infra.yp_dru.status import send_states


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(description='DynamicResource updater workload')
    parser.add_argument('--statefile', required=True, help='Path to file where state is stored and loaded from')
    parser.add_argument('--infosfile', required=True, help='Path to file where cached spec is stored and loaded from')
    parser.add_argument('--loglevel',
                        choices=('DEBUG', 'INFO', 'WARNING', 'ERROR'),
                        default='INFO',
                        help='Log level to use')
    parser.add_argument('--logfile', help='File to log into. If not specified, will log to stderr')
    parser.add_argument('--logsize', help='Log file size for rotating file handler', type=int, default=(1 << 22))
    parser.add_argument('--logcount', help='Log files to keep', type=int, default=14)

    hostname = socket.getfqdn()
    match = re.search('\.([a-z0-9-]+)\.yp-(c|test)\.yandex\.net$', hostname, re.IGNORECASE)
    if match:
        parser.add_argument('--proxy', default=f'https://{match.group(1)}.drcp.yandex-team.ru', help='DRCP address')
    else:
        parser.add_argument('--proxy', required=True, help='DRCP address')

    porto_name = os.getenv('PORTO_NAME', '')
    match = re.match('^ISS-AGENT--([a-zA-Z0-9_-]+)/', porto_name)
    if match:
        parser.add_argument('--pod', default=match.group(1), help='Current POD id')
    else:
        parser.add_argument('--pod', required=True, help='Current POD id')
    return parser.parse_args()


def get_last_revision(path: str) -> Optional[str]:
    try:
        with open(path, 'r') as f:
            data = json.load(f)
            send_states(data['resources'])
            return data['revision']
    except Exception as e:
        logging.getLogger('dru').exception(f'failed to get last revision, assume none: {e}')
        return None


def get_cached_infos(path: str) -> Optional[dict]:
    try:
        with open(path, 'r') as f:
            data = json.load(f)
            return data
    except Exception as e:
        logging.getLogger('dru').exception(f'failed to get last revision, assume none: {e}')
        return None


async def updater_loop(
    loop: asyncio.AbstractEventLoop,
    revision: Optional[str],
    pod: str,
    proxy: str,
    statefile: str,
    infosfile: str,
) -> None:
    while True:

        try:
            result = await fetch_resources_info(proxy, pod, revision)
        except (asyncio.TimeoutError, NoChanges):
            logging.getLogger('dru').info("no new data received, will use cached infos for check")
            result = get_cached_infos(infosfile)
            if result is None:
                logging.getLogger('dru').debug("cached infos are not available, will try to fetch again")
                continue
        except Exception as e:
            logging.getLogger('dru').exception(f"failed to fetch info from DRCP: {e}")
            await asyncio.sleep(5)
            continue
        else:
            with open(infosfile, 'w') as f:
                json.dump(result, f)

        if not result:
            logging.getLogger('dru').info("info from DRCP is empty, probably DRU will die soon")
            await asyncio.sleep(5)
            continue

        revision = result['revision']
        resources = result['resources']

        try:
            states = await process_all(statefile, resources, revision)
        except Exception as e:
            logging.getLogger('dru').exception(f"failed to update resources: {e}")
            await asyncio.sleep(5)
            continue

        send_states(states)
        break  # temporarily, because pod_agent do not read stdout otherwise


def main():
    args = parse_args()

    if not args.logfile:
        logging.basicConfig(level=args.loglevel)
    else:
        log = logging.getLogger('')
        handler = logging.handlers.RotatingFileHandler(
            args.logfile,
            maxBytes=args.logsize,
            backupCount=args.logcount,
        )
        handler.setFormatter(logging.Formatter(
            fmt='%(asctime)s.%(msecs)003d [%(levelname)-1s] [%(name)-21s]   %(message)s',
            datefmt='',
        ))
        log.setLevel(args.loglevel)
        log.addHandler(handler)

    revision = get_last_revision(args.statefile)
    loop = asyncio.get_event_loop()
    loop.run_until_complete(updater_loop(
        loop,
        revision,
        args.pod,
        args.proxy,
        args.statefile,
        args.infosfile,
    ))
