import os
import time
import shlex
import signal
import logging
import asyncio
from typing import Optional, Tuple


async def check_hashsum(path: str, method: str, checksum: str) -> bool:
    log = logging.getLogger('dru.check')
    if not os.path.exists(path):
        log.debug("path %r does not exist", path)
        return False

    qpath = shlex.quote(path)
    cmd = f"LC_COLLATE=C find {qpath} -type f -print0 | sort -z | xargs -0 cat | {method} -b | awk '{{print $1}}'"

    def restore_sig_handler():
        signal.signal(signal.SIGHUP, signal.SIG_DFL)

    process = await asyncio.create_subprocess_shell(
        cmd,
        preexec_fn=restore_sig_handler,
        stdout=asyncio.subprocess.PIPE,
        stderr=asyncio.subprocess.PIPE,
    )

    stdout, stderr = await process.communicate()
    if process.returncode != 0:
        log.warning('path %r checksum failed with %s: %s', path, process.returncode, stderr)
        return False

    return stdout.decode(errors='ignore').strip() == checksum


def get_check_fun(checksum: Optional[str]) -> Tuple[Optional[str], Optional[str]]:
    if not checksum or checksum.startswith('EMPTY:'):
        return None, None
    elif checksum.startswith('MD5:'):
        return 'md5sum', checksum[4:]
    elif checksum.startswith('SHA256:'):
        return 'sha256sum', checksum[7:]
    else:
        raise Exception(f"Don't know how to check hashsum {checksum!r}")


async def check(
    target_path: str,
    check_options: Optional[dict],
    last_check_time: Optional[float]
) -> Tuple[bool, float]:
    if check_options is None:
        return True, time.time()

    if last_check_time and (
        not check_options['check_period_ms']
        or 1000 * (time.time() - last_check_time) < check_options['check_period_ms']
    ):
        return True, last_check_time

    method, checksum = get_check_fun(check_options.get('checksum'))

    if method is None or checksum is None:
        return True, time.time()

    result = await check_hashsum(path=target_path, method=method, checksum=checksum)
    return result, time.time()
