import os
import json
import shutil
import logging
import asyncio
from typing import List, Optional

import aiohttp


async def download_raw(url: str, target_path: str) -> str:
    if os.path.isdir(target_path):
        shutil.rmtree(target_path)

    with open(target_path, 'w') as f:
        f.write(url[4:])  # strip "raw:"

    return target_path


async def download_http(url: str, target_path: str) -> str:
    log = logging.getLogger('dru.http')

    async with aiohttp.ClientSession() as session:
        log.debug(f"fetching {url}")
        try:
            if os.path.isdir(target_path):
                shutil.rmtree(target_path)

            async with session.get(url) as resp:
                if not (200 <= resp.status < 300):
                    raise Exception(f"Resource {url!r} is not available: {resp.status}")
                with open(target_path, 'wb') as f:
                    async for chunk in resp.content.iter_any():
                        f.write(chunk)

        except aiohttp.client_exceptions.ClientConnectionError as e:
            raise Exception(f"Resource {url!r} is not available: {e}")

    return target_path


async def download_rbtorrent(
    url: str,
    target_path: str,
    allow_deduplication: bool,
    max_download_speed: Optional[int],
) -> str:
    if os.path.isfile(target_path) or os.path.islink(target_path):
        os.unlink(target_path)

    if not os.path.exists(target_path):
        os.mkdir(target_path)

    additional_args = []
    if allow_deduplication:
        additional_args.append('--deduplicate=Hardlink')
    if max_download_speed:
        additional_args.append(f'--max-dl-speed={max_download_speed}mbps')

    download_proc = await asyncio.create_subprocess_exec(
        '/usr/local/bin/sky', 'get', *additional_args, '-d', target_path, url,
        stderr=asyncio.subprocess.PIPE,
    )
    files_proc = await asyncio.create_subprocess_exec(
        '/usr/local/bin/sky', 'files', '--json', url,
        stdout=asyncio.subprocess.PIPE,
        stderr=asyncio.subprocess.PIPE,
    )
    files_comm, download_comm = await asyncio.gather(files_proc.communicate(), download_proc.communicate(), return_exceptions=True)

    _, stderr = download_comm  # type: ignore
    if download_proc.returncode != 0:
        raise Exception(f"Cannot get resource {url!r} ({download_proc.returncode}): {stderr!r}")

    stdout, stderr = files_comm  # type: ignore
    if files_proc.returncode != 0:
        raise Exception(f"Cannot get resource {url!r} files ({files_proc.returncode}): {stderr!r}")

    try:
        files = json.loads(stdout)
    except Exception as e:
        raise Exception(f"Cannot parse resource {url!r} files list: {e}")

    if len(files) == 1:
        return os.path.join(target_path, next(iter(files))['name'])
    else:
        return target_path


async def download(
    urls: List[str],
    target_path: str,
    allow_deduplication: bool,
    max_download_speed: Optional[int] = None,
) -> str:
    log = logging.getLogger('dru.dwnl')
    exceptions = []
    for url in urls:
        try:
            if url.startswith('raw:'):
                return await download_raw(url, target_path)
            elif url.startswith('http://') or url.startswith('https://'):
                return await download_http(url, target_path)
            elif url.startswith('rbtorrent:'):
                return await download_rbtorrent(
                    url,
                    target_path,
                    allow_deduplication=allow_deduplication,
                    max_download_speed=max_download_speed,
                )
            else:
                raise Exception(f"Unsupported url scheme: {url}")
        except Exception as e:
            log.exception(f"download {url!r} failed with: {e}")
            exceptions.append(str(e))

    raise Exception("All download attempts failed:\n%s" % ('\n'.join(exceptions),))
