import signal
import logging
import asyncio
from typing import Optional

import aiohttp


async def notify_http(action: dict) -> bool:
    log = logging.getLogger('dru.notify')
    url = action.get('url')
    expected_answer = action.get('expected_answer')
    if url is None:
        return True

    async with aiohttp.ClientSession() as session:
        log.debug(f'fetching {url}')
        async with session.get(url) as resp:
            if not (200 <= resp.status < 300):
                log.info("notify about %r failed: %s", url, resp.status)
                return False

            if expected_answer is not None:
                answer = await resp.text()
                if answer != expected_answer:
                    log.info(
                        "notify hook answer doesn't match expected one, got: %r",
                        (answer[:97] + '...') if len(answer) > 100 else answer,
                    )
                    return False

    return True


async def notify_exec(action: dict) -> bool:
    log = logging.getLogger('dru.notify')

    def restore_sig_handler():
        signal.signal(signal.SIGHUP, signal.SIG_DFL)

    cmd = action.get('command_line')

    if not cmd:
        return True

    process = await asyncio.create_subprocess_shell(
        cmd,
        stdout=asyncio.subprocess.PIPE,
        stderr=asyncio.subprocess.PIPE,
        preexec_fn=restore_sig_handler,
    )

    stdout, stderr = await process.communicate()
    if process.returncode != 0:
        log.info('notify %r failed with %s: %s', cmd, process.returncode, stderr)
        return False

    stdout_str = stdout.decode(errors='replace').rstrip('\r\n')
    expected_stdout = action.get('expected_answer')
    if expected_stdout is not None and expected_stdout != stdout_str:
        log.info(
            "notify hook answer doesn't match expected one, got: %r",
            (stdout_str[:97] + '...') if len(stdout_str) > 100 else stdout_str,
        )
        return False

    return True


async def notify(http_action: Optional[dict], exec_action: Optional[dict]) -> bool:
    if http_action is not None:
        return await notify_http(http_action)
    elif exec_action is not None:
        return await notify_exec(exec_action)
    else:
        return True
