import sys
import enum
import json
from typing import Optional


class Process(enum.Enum):
    INSTALL = 'install'
    UPGRADE = 'upgrade'
    CHECK = 'check'
    REMOVAL = 'remove'


class Action(enum.Enum):
    DOWNLOAD = 'download'
    CHECK = 'check'
    SWITCH = 'switch'
    NOTIFY = 'notify'
    REMOVE = 'remove'


class State(enum.Enum):
    IN_PROGRESS = 'in_progress'
    READY = 'ready'
    ERROR = 'error'


def send_action_status(
    resource_id: str,
    action: Action,
    process: Process,
    revision: Optional[str],
    state: State,
    resource_state: dict,
    reason: Optional[str] = None,
) -> None:
    msg = {
        'kind': 'event',
        'resource_id': resource_id,
        'action': action.value,
        'process': process.value,
        'revision': revision,
        'mark': resource_state.get('mark'),
    }
    if state == State.IN_PROGRESS:
        resource_state['reason'] = msg['reason'] = reason
        resource_state['in_progress'] = msg['in_progress'] = True
        resource_state['ready'] = resource_state['error'] = msg['ready'] = msg['error'] = False
    elif state == State.READY:
        resource_state['reason'] = msg['reason'] = reason
        resource_state['ready'] = msg['ready'] = True
        resource_state['in_progress'] = resource_state['error'] = msg['in_progress'] = msg['error'] = False
    elif state == State.ERROR:
        resource_state['reason'] = msg['reason'] = reason
        resource_state['error'] = msg['error'] = True
        resource_state['in_progress'] = resource_state['ready'] = msg['in_progress'] = msg['ready'] = False

    resource_state['revision'] = revision

    print(json.dumps(msg), flush=True)
    print(json.dumps(msg), flush=True, file=sys.stderr)


def send_states(states: dict) -> None:
    msg = {
        'kind': 'state',
        'states': [{
            'resource_id': resource_id,
            'revision': state.get('revision'),
            'in_progress': state['in_progress'],
            'ready': state['ready'],
            'error': state['error'],
            'reason': state['reason'],
            'mark': state.get('mark'),
        } for resource_id, state in states.items()],
    }
    print(json.dumps(msg), flush=True)
    print(json.dumps(msg), flush=True, file=sys.stderr)
