import os
import json
import shutil
import logging
import asyncio
from typing import Optional, List, Tuple, Awaitable

import aiohttp
from yarl import URL

from .check import check
from .notify import notify
from .download import download
from .status import Process, Action, State, send_action_status


# state: {
#   'revision': revision,
#   'symlink': destination,
#   'path': storage_dir.subpath,
#   'base_path': storage_dir.subpath,
#   'http_action': { 'url': url, 'expected_answer': expected_answer },
#   'exec_action': { 'command_line': command_line, 'expected_answer': expected_answer },
#   'last_check_time': timestamp_in_float_seconds,
#   'in_progress': in_progress,
#   'ready': ready,
#   'error': error,
#   'reason': reason,
#   'revs': [
#       {
#           'path': storage_dir.subpath,
#           'base_path': storage_dir.subpath,
#           'revision': revision,
#           'urls': urls,
#           'dirty': bool,
#           'aliases': [symlink_path],
#       },
#   ],
# }


class NoChanges(Exception):
    pass


async def fetch_resources_info(
    base_url: str,
    pod_id: str,
    revision: Optional[str] = None
) -> dict:
    log = logging.getLogger('dru')
    url = URL(base_url)
    url = url.join(URL(pod_id) if revision is None else URL(f'{pod_id}/{revision}'))

    # We intentionally recreate session each time because
    # server will close connection to protect itself from all
    # the clients binding to the same instance forever
    async with aiohttp.ClientSession() as session:
        log.info(f"fetching {url}")
        try:
            async with session.get(url) as resp:
                if not (200 <= resp.status < 300):
                    raise RuntimeError(f"DRCP pod is not available: {resp.status}")

                return await resp.json()
        except aiohttp.ServerDisconnectedError:
            raise NoChanges()
        except aiohttp.client_exceptions.ClientConnectionError as e:
            raise RuntimeError(f"DRCP is not available: {e}")


class ResourceUpdateException(Exception):
    def __init__(self, resource_id, revision, message):
        self.resource_id = resource_id
        self.revision = revision
        super().__init__(f"Failed to update resource {resource_id!r} to revision {revision}: {message}")


async def switch_resource(state: dict, info: dict) -> None:
    log = logging.getLogger('dru.switch')
    resource_id = info['id']
    revision = info['revision']
    process = Process.INSTALL if state.get('revision', revision) == revision else Process.UPGRADE

    log.info("will %s resource %r (%r -> %r)", process, resource_id, state.get('revision', revision), revision)

    send_action_status(
        resource_id=resource_id,
        action=Action.DOWNLOAD,
        process=process,
        revision=revision,
        state=State.IN_PROGRESS,
        resource_state=state,
    )

    try:
        storage_dir = info['storage_options']['storage_dir']
        target_path = os.path.join(storage_dir, str(revision))
        cached_count = info['storage_options'].get('cached_revisions_count', 2)

        if cached_count < 1:
            raise ResourceUpdateException(
                resource_id,
                revision,
                f"cached_revisions_count has invalid value {cached_count}"
            )

        if not os.path.exists(storage_dir):
            raise ResourceUpdateException(
                resource_id,
                revision,
                f"storage dir {storage_dir!r} does not exist"
            )

        revs = state.setdefault('revs', [])

        revs_item = None
        for item in revs:
            if item.get('urls') == info['urls']:
                item['revision'] = revision
                if target_path != item['path'] and target_path not in item.get('aliases', []):
                    item.setdefault('aliases', []).append(target_path)
                state['path'] = item['path']
                state['base_path'] = item['base_path']
                revs.remove(item)
                revs.insert(0, item)
                revs_item = item

                break
        else:
            revs_item = {
                'revision': revision,
                'path': target_path,
                'urls': info['urls'],
                'allow_deduplication': info['storage_options'].get('allow_deduplication', False),
                'max_download_speed': info['storage_options'].get('max_download_speed', None),
                'mark': info.get('mark'),
                'base_path': None,
                'dirty': True,
                'aliases': [],
            }
            revs.insert(0, revs_item)
            state['path'] = target_path
        revs_item['mark'] = info.get('mark')
        log.info("resource %r done", resource_id)

    except Exception as e:
        log.exception("resource %r failed: %s", resource_id, e)
        send_action_status(
            resource_id=resource_id,
            action=Action.DOWNLOAD,
            process=process,
            revision=revision,
            state=State.ERROR,
            reason=str(e),
            resource_state=state,
        )
        return

    try:
        cleanup_old_copies(state, cached_count)
    except Exception as e:
        log.info("failed to remove old path: %s", e)
        send_action_status(
            resource_id=resource_id,
            action=Action.CHECK,
            process=Process.CHECK,
            revision=revision,
            state=State.ERROR,
            reason=str(e),
            resource_state=state,
        )
        return

    # we we start from the most recent revision in cache and try to apply it until we try all N
    for _ in range(len(revs)):
        if not await try_apply_resource(state, info):
            revs[0]['dirty'] = True
            revs_item = revs.pop(0)
            revs.append(revs_item)
        else:
            return


async def try_apply_resource(state: dict, info: dict) -> bool:
    log = logging.getLogger('dru.apply')
    action = Action.DOWNLOAD
    revs_item = state['revs'][0]
    revision = revs_item['revision']
    process = Process.INSTALL if state.get('revision', revision) == revision else Process.UPGRADE

    try:
        if revs_item['dirty']:
            revs_item['base_path'] = state['base_path'] = await download(
                revs_item['urls'],
                revs_item['path'],
                allow_deduplication=revs_item.get('allow_deduplication', False),
                max_download_speed=revs_item.get('max_download_speed', None),
            )
            action = Action.CHECK
            send_action_status(
                resource_id=info['id'],
                action=action,
                process=process,
                revision=revision,
                state=State.IN_PROGRESS,
                resource_state=state,
            )

            check_result, check_time = await check(
                revs_item['path'],
                info['storage_options'].get('verification'),
                None,
            )
            if not check_result:
                raise ResourceUpdateException(
                    info['id'],
                    revision,
                    "resource just downloaded is corrupted",
                )

            state['last_check_time'] = check_time
            revs_item['dirty'] = False

        state['mark'] = revs_item.get('mark')
        action = Action.SWITCH
        send_action_status(
            resource_id=info['id'],
            action=Action.SWITCH,
            process=process,
            revision=revision,
            state=State.IN_PROGRESS,
            resource_state=state,
        )

        switch_path(state, info)

        action = Action.NOTIFY
        await notify_service(state, info, process)

        send_action_status(
            resource_id=info['id'],
            action=action,
            process=process,
            revision=revision,
            state=State.READY,
            resource_state=state,
        )

        return True

    except Exception as e:
        send_action_status(
            resource_id=info['id'],
            action=action,
            process=process,
            revision=revision,
            state=State.ERROR,
            reason=str(e),
            resource_state=state,
        )
        log.exception("resource %r revision %s %s failed: %s", info['id'], revision, action.name, e)

        return False


async def check_resource(state: dict, info: dict) -> None:
    log = logging.getLogger('dru.check')
    verification = info['storage_options'].get('verification')
    # At this point only 'revision' is guaranteed to present in state,
    # any other field may be lost
    target_path = state.get('path')
    last_check_time = state.get('last_check_time')

    log.info("will check resource %r", info['id'])
    ready = state.get('ready', False)  # switch to in_progress will clean this flag, so check it early

    send_action_status(
        resource_id=info['id'],
        action=Action.CHECK,
        process=Process.CHECK,
        revision=state['revision'],
        state=State.IN_PROGRESS,
        resource_state=state,
    )
    try:
        log.debug('will check for %r, verification = %r', target_path, verification)

        if not state.get('base_path') or not state.get('path'):
            log.debug('resource is not ready, probably not downloaded yet')
            raise ResourceUpdateException(info['id'], state['revision'], "resource not downloaded, will download")

        check_result, check_time = await check(target_path, verification, last_check_time)  # type: ignore
        log.debug('check ended with %r, last_check_time = %r', check_result, check_time)
        if check_result:
            state['last_check_time'] = check_time

            if not ready:
                send_action_status(
                    resource_id=info['id'],
                    action=Action.SWITCH,
                    process=Process.CHECK,
                    revision=state['revision'],
                    state=State.IN_PROGRESS,
                    resource_state=state,
                )

                switch_path(state, info)
                await notify_service(state, info, Process.CHECK)

            send_action_status(
                resource_id=info['id'],
                action=Action.CHECK,
                process=Process.CHECK,
                revision=state['revision'],
                state=State.READY,
                resource_state=state,
            )
            log.info("resource %r done", info['id'])
            return
        raise ResourceUpdateException(info['id'], state['revision'], "check failed, will redownload")
    except Exception as e:
        log.exception("resource %r failed: %s", info['id'], e)
        send_action_status(
            resource_id=info['id'],
            action=Action.CHECK,
            process=Process.CHECK,
            revision=state['revision'],
            state=State.ERROR,
            reason=str(e),
            resource_state=state,
        )

    await switch_resource(state, info)


def delete(path: str) -> None:
    if os.path.islink(path) or os.path.isfile(path):
        os.unlink(path)
    elif os.path.exists(path):
        shutil.rmtree(path)


def remove_path(state: dict, path_item: str) -> None:
    path = state.get(path_item)
    if path is not None:
        delete(path)
        del state[path_item]


def switch_path(state: dict, info: dict) -> None:
    old_dest_path = state.get('symlink')
    state['symlink'] = dest_path = info['storage_options']['destination'].rstrip('/')
    temp_dest_path = f'{dest_path}.new'
    aliases = set(state['revs'][0].get('aliases', []))

    for symlink in aliases:
        delete(symlink)
        os.symlink(state['path'], symlink)

    delete(temp_dest_path)
    os.symlink(state['base_path'], temp_dest_path)

    state['aliases'] = list(aliases)

    os.rename(temp_dest_path, dest_path)

    if old_dest_path is not None and old_dest_path != dest_path and os.path.islink(old_dest_path):
        os.unlink(old_dest_path)


def cleanup_old_copies(state: dict, copies: int):
    while len(state['revs']) > copies:
        paths = set(state['revs'][-1].setdefault('aliases', [])) | {state['revs'][-1]['path']}

        for path in paths:
            delete(path)

        state['revs'].pop()


async def notify_service(state: dict, info: dict, process: Process) -> None:
    send_action_status(
        resource_id=info['id'],
        action=Action.NOTIFY,
        process=process,
        revision=info['revision'],
        state=State.IN_PROGRESS,
        resource_state=state,
    )

    notified = await notify(info['storage_options'].get('http_action'),
                            info['storage_options'].get('exec_action'))

    if notified:
        # Update actions in state so in case of resource removal we know last action to call
        state['http_action'] = info['storage_options'].get('http_action')
        state['exec_action'] = info['storage_options'].get('exec_action')
    else:
        raise ResourceUpdateException(
            info['id'],
            state['revision'],
            'notify failed',
        )


async def remove_resource(resource_id: str, state: dict) -> Tuple[str, Optional[dict]]:
    log = logging.getLogger('dru.remove')
    log.info("removing resource %r", resource_id)

    send_action_status(
        resource_id=resource_id,
        action=Action.REMOVE,
        process=Process.REMOVAL,
        revision=None,
        state=State.IN_PROGRESS,
        resource_state=state,
    )
    action = Action.REMOVE
    try:
        remove_path(state, 'symlink')
        while state['revs']:
            path = state['revs'][-1]['path']
            delete(path)
            state['revs'].pop()

        del state['path']

        action = Action.NOTIFY
        send_action_status(
            resource_id=resource_id,
            action=Action.NOTIFY,
            process=Process.REMOVAL,
            revision=None,
            state=State.IN_PROGRESS,
            resource_state=state,
        )
        notified = await notify(state.get('http_action'), state.get('exec_action'))
        if notified:
            state.pop('http_action', None)
            state.pop('exec_action', None)
            send_action_status(
                resource_id=resource_id,
                action=action,
                process=Process.REMOVAL,
                revision=None,
                state=State.READY,
                resource_state=state,
            )
            log.info("resource %r removed", resource_id)
            return resource_id, None
        else:
            raise ResourceUpdateException(resource_id, None, "notify failed, will retry later")
    except Exception as e:
        log.exception("remove resource %r failed: %s", resource_id, e)
        send_action_status(
            resource_id=resource_id,
            action=action,
            process=Process.REMOVAL,
            revision=None,
            state=State.ERROR,
            reason=str(e),
            resource_state=state,
        )
        return resource_id, state


async def process_resource(state: dict, info: dict) -> Tuple[str, dict]:
    resource_id = info['id']

    if state.get('revision') == info['revision']:
        await check_resource(state, info)
        return resource_id, state

    await switch_resource(state, info)

    return resource_id, state


async def process_all(state_path: str, infos: List[dict], revision: str) -> dict:
    log = logging.getLogger('dru')
    try:
        states = json.load(open(state_path, 'r'))['resources']
        if not isinstance(states, dict):
            log.info("state file is corrupted, starting from scratch")
            states = {}
    except Exception as e:
        log.info("old state is not available: %s", e)
        states = {}

    infos_dict = {info['id']: info for info in infos}
    jobs: List[Awaitable[Tuple[str, Optional[dict]]]] = []
    for resource_id, state in states.items():
        if resource_id not in infos_dict:
            jobs.append(remove_resource(resource_id, state))

    for resource_id, info in infos_dict.items():
        state = states.setdefault(resource_id, {})
        if not isinstance(state, dict):
            state = {}
        jobs.append(process_resource(state, info))

    for job in asyncio.as_completed(jobs):
        try:
            resource_id, state = await job
        except Exception as e:
            log.info("failed to update: %s", e)
            continue

        states.pop(resource_id, None)
        if state is not None:
            states[resource_id] = state

    new_state_path = f'{state_path}.new'
    with open(new_state_path, 'w') as f:
        json.dump({
            'revision': revision,
            'resources': states,
        }, f)
    os.rename(new_state_path, state_path)
    return states
