import os
import json
import jinja2
import library.python.resource as resource

from status import combine_record_status_info

from flask import Flask, Response, render_template, request, url_for, jsonify
from flask_bootstrap import Bootstrap
from infra.yp_quota_distributor.app.qloud import qloud_blueprint
from infra.yp_quota_distributor.app.gencfg import gencfg_blueprint
from infra.yp_quota_distributor.app.hosts import hosts_blueprint

from infra.yp_quota_distributor.lib.mongo import mongo_db
from bson.objectid import ObjectId
from infra.yasm.unistat import global_unistat

from infra.yp_quota_distributor.lib.constants import (
    ERROR_STATUS_API_WITHOUT_REQUEST_ID,
    ERROR_INFO
)
from infra.yp_quota_distributor.lib.common import increase_unistat_signal_value


def _load_template_from_resources(name):
    return resource.find("resfs/file/{}".format(name))


COOKIE_SECRET_KEY = os.environ.get('COOKIE_SECRET_KEY', 'secret')
app = Flask(__name__, static_url_path='')

Bootstrap(app)
app.config['BOOTSTRAP_SERVE_LOCAL'] = True
app.config['SECRET_KEY'] = COOKIE_SECRET_KEY
app.jinja_env.loader = jinja2.FunctionLoader(_load_template_from_resources)

app.register_blueprint(gencfg_blueprint, url_prefix='/gencfg/')
app.register_blueprint(qloud_blueprint, url_prefix='/qloud/')
app.register_blueprint(hosts_blueprint, url_prefix='/hosts/')

signals = {
    'request-exceptions-number-total':
    global_unistat.create_float('request-exceptions-number-total',
                                priority=200,
                                always_visible=True),
    'approve-exceptions-number-total':
    global_unistat.create_float('approve-exceptions-number-total',
                                priority=200,
                                always_visible=True),
    'dump-info-exceptions-number-total':
    global_unistat.create_float('dump-info-exceptions-number-total',
                                priority=200,
                                always_visible=True),
    'old-dispenser-requests-exceptions-number-total':
    global_unistat.create_float(
        'old-dispenser-requests-exceptions-number-total',
        priority=200,
        always_visible=True)
}

RECORD_STATUS_INFO = combine_record_status_info()


@app.route('/')
def index():
    return render_template('templates/index.html')


@app.route('/unistat/', methods=['GET'])
def unistat():
    for name, value in signals.iteritems():
        val = increase_unistat_signal_value(name, 0)
        value.push(val)
    return jsonify(json.loads(global_unistat.to_json()))


@app.route('/static/<path:path>')
def send_static(path):
    response = resource.find('/static/{}'.format(path))
    if path.endswith('.js'):
        mimetype = "application/javascript"
    elif path.endswith('.css'):
        mimetype = "text/css"
    else:
        mimetype = "text/plain"
    return Response(response=response, mimetype=mimetype)


@app.route('/status/', methods=['GET'])
def status():
    record_id = request.args.get('record_id')
    if not record_id:
        return render_template('error.html', text=ERROR_STATUS_API_WITHOUT_REQUEST_ID)

    def handle_collection(collection, status, css_class):
        record = collection.find_one({"_id": ObjectId(record_id)})
        if record:
            # Old status version
            error_text = ""
            if "error_text" in record:
                error_text = record["error_text"]
            if 'stop' in record or error_text:
                css_class = "isa_error"
                status_info = ERROR_INFO
            else:
                status_info = RECORD_STATUS_INFO["{}.{}".format(
                    record['type'], record['showState']
                    if 'showState' in record else record['state'])]
            if 'status' in record:
                css_class = record['status']['css_class']
                status_info += ". " + record['status']['text']

            return render_template(
                    'templates/status.html',
                    status_info=status_info,
                    css_class=css_class,
                    status=status,
                    ticket=record['ticket'] if 'ticket' in record else None,
                    main_url=url_for('index'),
                    error_text=error_text.replace("\n", "<br>"),
                    problem=record.get('problem', ''),
                    contact_login=record.get("contact_login", ""))
        else:
            return None

    template = handle_collection(mongo_db["requests_queue"], "PROCESSING", "isa_info")
    if not template:
        template = handle_collection(mongo_db["completed_requests"], "DONE", "isa_success")

    if template:
        return template

    return render_template('templates/error.html', text=ERROR_STATUS_API_WITHOUT_REQUEST_ID)
