# -*- coding: utf-8 -*-

import requests
from flask_wtf import FlaskForm
from wtforms import StringField, SubmitField
from wtforms.validators import DataRequired

from infra.yp_quota_distributor.lib.common import (
    get_service_dump_by_name
)

from infra.yp_quota_distributor.lib.constants import (
    TOKEN,
    NANNY_URL,
    GENCFG_GROUPS,
    NANNY_SERVICES,
    ABC_REMAPPING
)


def get_fields_from_form(form):
    abc_info = str(form.abc_service.data).encode("utf-8").strip('/').split('/')[-1]
    gencfg_groups = str(form.gencfg_groups.data).replace(" ", "").strip(", ").split(",")
    if gencfg_groups == [""]:
        gencfg_groups = []
    nanny_info = str(form.nanny_services.data).replace(" ", "").strip(", ").split(",")
    nanny_services = [data.strip('/').split('/')[-1] for data in nanny_info if data]
    return {"abc_info": abc_info, "gencfg_groups": gencfg_groups, "nanny_services": nanny_services}


class GencfgQoutaForm(FlaskForm):
    abc_service = StringField(
        'ABC service id or slug (Example: "gencfg" or "1175")',
        validators=[DataRequired()])
    nanny_services = StringField('Nanny service ids (Example: "news_nginx,yp_quota_distributor,...")')
    gencfg_groups = StringField('Gencfg groups (Example: "MAN_AB_BALANCER,ALL_GENCFG_CLICKHOUSE,...")')
    submit = SubmitField('Issue the quota')

    def __init__(self, owner, *args, **kwargs):
        super(GencfgQoutaForm, self).__init__(*args, **kwargs)
        self.owner = owner

    def validate(self):

        def validate_abc_service(abc_info, owner):
            valid = True
            abc_dump = get_service_dump_by_name(abc_info, ABC_REMAPPING)
            if not abc_dump:
                self.abc_service.errors.append("Not ABC service: '{}'".format(str(abc_info)))
                return False

            if owner not in abc_dump["members"]:
                self.abc_service.errors.append("You cannot choose ABC service: '{}'".format(str(abc_info)))
                return False

            return valid

        def validate_gencfg_groups(groups, owner):
            owners_gencfg_groups = []
            gencfg_dump = get_service_dump_by_name(owner, GENCFG_GROUPS)
            if gencfg_dump:
                owners_gencfg_groups = gencfg_dump["groups"]
            valid = True
            for group in groups:
                if not group:
                    continue
                if group not in owners_gencfg_groups:
                    valid = False
                    self.gencfg_groups.errors.append(
                        "Not valid group '{}' or you are not the owner".format(
                            group))
            return valid

        def validate_nanny_services(services):
            nanny_session = requests.Session()
            nanny_session.headers['Authorization'] = 'OAuth {}'.format(TOKEN)

            valid = True
            for service in services:
                if not service or not get_service_dump_by_name(service, NANNY_SERVICES):
                    continue

                try:
                    response = nanny_session.get('{}/v2/services/{}'.format(NANNY_URL, service), timeout=30)
                except:
                    valid = False
                    self.nanny_services.errors.append("Not valid nanny service: '{}'".format(service))

                if response.status_code != requests.codes.ok:
                    valid = False
                    self.nanny_services.errors.append("Not valid nanny service: '{}'".format(service))
            return valid

        fields = get_fields_from_form(self)
        valid = super(GencfgQoutaForm, self).validate()
        valid = validate_abc_service(fields["abc_info"], self.owner) and valid
        valid = validate_gencfg_groups(fields["gencfg_groups"], self.owner) and valid
        valid = validate_nanny_services(fields["nanny_services"]) and valid
        return valid
