# -*- coding: utf-8 -*-

import requests
import logging
import copy
import retrying

from infra.yp_quota_distributor.lib.mongo import mongo_db

from infra.yp.account_estimation import (
    customize_quota,
    combine_gencfg_and_nanny_groups,
    get_not_yp_nanny_services,
    get_resources_for_groups_with_tag_by_instances,
    amputate_resources_for_infrastructure
)
from infra.yp_quota_distributor.lib.constants import (
    HOST,
    TOKEN,
    ABC_URL,
    ABC_RESOURCES_TEMPLATE,
    ROBOT_GENCFG,
    VALID_MASTER_GROUPS,
    STARTREK_URL,
    GENCFG_GROUPS,
    ABC_REMAPPING
)

from status import (
    Status,
    StopReason,
    STOP_REASON_INFO,
)

from infra.yp_quota_distributor.lib.common import (
    is_abc_service_banned,
    get_ticket_for_banned_service,
    check_guest_group,
    get_master_group,
    is_valid_slave,
    get_quota_table_by_accounts,
    get_service_dump_by_name,
    update_collection_record
    # convert_account_to_dispenser_resources,
    # convert_resources_to_minimal_units
)

from infra.yp_quota_distributor.lib.record_handlers import (
    request_quota_in_abc_service,
    YP_DC_mapping,
    YP_Distributor
)

from infra.yp_quota_distributor.lib.common import startrek_api

from yp.client import YpClient

MAX_CPU_QUOTA = 3 * (10 ** 5)


logger = logging.getLogger(__name__)


def build_yp_client(cluster=None):
    return YpClient(cluster + '.yp.yandex-team.ru:8090', config={"token": TOKEN})


# TODO(atridis): Handle error response from startrek
# don't close the ticket until all gencfg groups will not be deleted
@retrying.retry(stop_max_attempt_number=1)
def create_and_close_ticket(record, table):
    abc_service_id = record["abc_service_id"]
    yp_nanny_services = record.get("yp_nanny_services", [])
    yp_gencfg_groups = record.get("yp_gencfg_groups", [])
    handled_gencfg_groups = record.get("handled_gencfg_groups", [])
    not_allowed_groups = record.get("not_allowed_groups", [])
    request_owner = record["yandex_login"]
    requested_gencfg_groups = record["gencfg_groups"]
    requested_nanny_services = record["nanny_services"]
    abc_slug = record["slug"]
    guest_groups_with_parents = record.get("guest_groups_with_parents", [])
    combined_gencfg_groups = [item[0] for item in record.get("unique_gencfg_groups_with_tags", [])]

    stop_reason = record.get("stop_reason", StopReason.NOTHING)

    session = requests.Session()
    session.headers['Authorization'] = 'OAuth {}'.format(TOKEN)
    assignee = ROBOT_GENCFG

    description = []
    description.append("You can see details below:")
    description.append("Owner of request: {}".format(request_owner))
    problem_resolver_login = "glebskvortsov"
    if requested_gencfg_groups:
        description.append("These gencfg groups were requested:")
        description.append(",".join(requested_gencfg_groups))
    if requested_nanny_services:
        description.append("These nanny services were requested:")
        description.append(",".join(requested_nanny_services))
    if combined_gencfg_groups:
        description.append("Gencfg groups obtained by merging requested groups and nanny services groups")
        description.append(",".join(combined_gencfg_groups))

    if table:
        description.append("<#" + table + "#>")

    description.append('\n'.join([
        "\n**Please note we've issued IO SSD and IO HDD quota for these groups.**",
        "**You have to use this IO SSD/IO HDD quota for your YP.Lite service.**",
        "**In case of any questions regarding IO quotas/bandwidth please contact glebskvortsov@**", ""]))

    if stop_reason != StopReason.NOTHING:
        description.append("**" + STOP_REASON_INFO[stop_reason] + "**")
        assignee = problem_resolver_login
    else:
        if yp_nanny_services:
            description.append("These nanny services are already in YP:")
            description.append(','.join(yp_nanny_services))

        if yp_gencfg_groups:
            description.append("These gencfg groups belong to services, which are already in YP:")
            description.append(','.join(yp_gencfg_groups))

        if handled_gencfg_groups:
            description.append("These gencfg groups are already handled:")
            handled_groups_with_tickets = []
            collection = mongo_db["unique_gencfg_groups"]
            for group in handled_gencfg_groups:
                cursor = collection.find_one({"_id": group})
                ticket_info = ""
                if "data" in cursor and "ticket" in cursor["data"]:
                    ticket_info = "(" + cursor["data"]["ticket"] + ")"
                handled_groups_with_tickets.append(group + ticket_info)
            description.append("\n".join(handled_groups_with_tickets))
        if not_allowed_groups:
            description.append("You are not the owner of these gencfg groups (gencfg info updated onсe in 24 hours) :")
            description.append(','.join(not_allowed_groups))

    description.append("Your YP resources in ABC service:")
    description.append(ABC_URL + ABC_RESOURCES_TEMPLATE.format(abc_slug))

    ticket_key = startrek_api.create_ticket(
        "YPRES",
        "Квота в YP для ABC сервиса: {}".format(abc_slug),
        assignee,
        '\n'.join([line.decode("utf-8") for line in description]),
        int(abc_service_id),
        ["yp_quota"],
        [request_owner]
    )["key"]

    if stop_reason == StopReason.MASTER_GROUP:
        comment = ["Среди сдаваемых групп есть мастер группы.",
                   "Пожалуйста, подтвердите, что после съезда вы полностью удалите указанные мастер группы со всеми слейвами и подождите ручного подтверждения."]
        startrek_api.add_comment_to_the_ticket(ticket_key, "\n".join(comment), [request_owner])
    elif stop_reason == StopReason.GUEST_GROUP:
        guest_parent_template = "Guest: {}, Parent: {}"
        comment = [
            "!!Квота не выдана!!",
            "В заявке присутсвуют гостевые группы. Сдавайте пожалуйста родительские группы",
            "Список гостевый групп и их родителей:",
        ]
        comment.extend([guest_parent_template.format(guest, parent) for guest, parent in guest_groups_with_parents])
        startrek_api.add_comment_to_the_ticket(ticket_key, "\n".join(comment), [request_owner])
        startrek_api.close_ticket(ticket_key, "won'tFix")
    elif stop_reason == StopReason.NOT_VALID_SLAVE_GROUP:
        comment = [
            "!!Квота не выдана!!",
            "Only slave groups from {} are automatically converted to YP quota.".format('/'.join(VALID_MASTER_GROUPS)),
            "To convert other master groups to quota please contact glebskvortsov@, sereglond@"
        ]
        startrek_api.add_comment_to_the_ticket(ticket_key, "\n".join(comment), [request_owner])
        startrek_api.close_ticket(ticket_key, "won'tFix")
    elif stop_reason == StopReason.LARGE_VALUES:
        comment = ["Ваша квота оказалась достаточно большой.",
                   "Пожалуйста, подождите ручного подтверждения.",
                   "Ссылка для ручного подтверждения (только для @glebskvortsov, @sereglond):",
                   HOST + "gencfg/approve/?record_id={}".format(record["_id"])]
        startrek_api.add_comment_to_the_ticket(ticket_key, "\n".join(comment))
    elif stop_reason == StopReason.BANNED_ABC_SERVICE:
        comment = [
            "!!Квота не выдана!!",
            "Для данного сервиса квота не может быть выделена. Подробности в этом тикете: {}".format(get_ticket_for_banned_service(abc_service_id))
        ]
        startrek_api.add_comment_to_the_ticket(ticket_key, "\n".join(comment), [request_owner])
        startrek_api.close_ticket(ticket_key, "won'tFix")
    else:
        startrek_api.start_ticket_progress(ticket_key)

    return ticket_key


# def request_quota_in_dispenser_service(record):
#     # tested
#     logger.info("request_quota_in_dispenser_service for record: '{}'".format(record["_id"]))
#     updating_fields = {}
#     dispenser_project_key = record["dispenser_project_key"]
#     account_per_dc = record["account_per_dc"]
#     groups_by_dc = record["groups_by_dc"]
#     dispenser_api = DispenserApi(host=DISPENSER_URL, cluster='clouds', oauth=TOKEN)
#
#     # tested
#     dc_success = {}
#     for dc in YP_DCs:
#         dc_success[dc] = False
#     if "dc_dispenser_request_successed" in record:
#         dc_success = record["dc_dispenser_request_successed"]
#
#     for key, value in account_per_dc.iteritems():
#         if key.lower() not in YP_DCs:
#             logger.error("Unsupported DC '{}' for record_id '{}'".format(key, record["_id"]))
#         location = key.upper()
#
#         # TODO(atridis): change account_per_dc to minimal units
#
#         resources = convert_account_to_dispenser_resources(convert_resources_to_minimal_units(value))
#
#         retries = 2
#         while retries and not dc_success[key]:
#             parents = client.get_project_parents(dispenser_project_key)
#             #  response = client.update_max_quota_delta(parents, resources, ["default", location], "yp")
#             if not response.ok:
#                 logger.error("Response from dispenser status={} headers={} text={}".format(
#                     response.status_code, str(response.headers).encode("utf-8", "strict"), response.text.encode("utf-8")))
#             else:
#                 dc_success[key] = True
#             retries -= 1
#
#     # tested
#     state_delta = True
#     for key, value in account_per_dc.iteritems():
#         state_delta = state_delta and dc_success[key]
#     updating_fields = {
#         "problem": " (dispenser replied 500 error, retrying)" if not state_delta else "",
#         "state": record["state"] + state_delta,
#         "dc_dispenser_request_successed": dc_success
#     }
#     update_collection_record(record["_id"], updating_fields)


def get_stop_reason_or_none(gencfg_groups_with_tags):
    for gencfg_group in gencfg_groups_with_tags:
        if get_master_group(gencfg_group[0]) and not is_valid_slave(gencfg_group[0]):
            return StopReason.NOT_VALID_SLAVE_GROUP, []

    guest_groups_with_parents = []
    for gencfg_group in gencfg_groups_with_tags:
        parent_group_for_guest = check_guest_group(gencfg_group[0])
        if parent_group_for_guest:
            guest_groups_with_parents.append([gencfg_group[0], parent_group_for_guest])
    if len(guest_groups_with_parents) > 0:
        return StopReason.GUEST_GROUP, guest_groups_with_parents

    return None, []


def estimate_account(record):
    logger.info("estimate_account for record: '{}'".format(record["_id"]))
    gencfg_groups = record["unique_gencfg_groups_with_tags"]

    account_per_dc, groups_by_dc = get_resources_for_groups_with_tag_by_instances(gencfg_groups, True)
    logger.info("Account per dc for record_id {}: {}".format(record["_id"], account_per_dc))
    account_per_dc = amputate_resources_for_infrastructure(account_per_dc)
    accounts = dict([key, value.__dict__] for key, value in account_per_dc.iteritems())

    honest_quota = copy.deepcopy(accounts)
    for key, value in honest_quota.iteritems():
        accounts[key] = customize_quota(value)

    updating_fields = {
        "groups_by_dc": groups_by_dc,
        "honest_quota": honest_quota,
        "account_per_dc": accounts,
    }
    logger.info("Estimated accounts for record_id {}: {}".format(record["_id"], accounts))

    if is_abc_service_banned(record["abc_service_id"]):
        updating_fields.update({"stop_reason": StopReason.BANNED_ABC_SERVICE})
        skip_stages_to_startrek(record, updating_fields)
        return

    for gencfg_group in gencfg_groups:
        if not get_master_group(gencfg_group[0]):
            updating_fields.update({"stop_reason": StopReason.MASTER_GROUP})
            skip_stages_to_startrek(record, updating_fields)
            return

    for dc, account in account_per_dc.iteritems():
        if accounts[dc]['cpu'] >= MAX_CPU_QUOTA:
            updating_fields.update({"stop_reason": StopReason.LARGE_VALUES})
            skip_stages_to_startrek(record, updating_fields)
            return

    updating_fields.update({"state": record["state"] + 1})
    update_collection_record(record["_id"], updating_fields)


def allocate_quota_and_create_ticket(record):
    logger.info("allocate_quota_and_create_ticket for record: '{}'".format(record["_id"]))

    account_per_dc = record.get("account_per_dc", {})
    dc_distributors = {}
    for dc in account_per_dc:
        dc_distributors[dc] = "abc" if YP_DC_mapping[dc] == YP_Distributor.abc else "abcd"

    table = get_quota_table_by_accounts(account_per_dc, distributors=dc_distributors)
    logger.debug("Table for record_id {}: {}".format(record["_id"], table))

    ticket_key = create_and_close_ticket(record, table.to_html())
    ticket_key = STARTREK_URL + ticket_key

    collection = mongo_db["unique_gencfg_groups"]
    for gencfg_group in record.get("unique_gencfg_groups_with_tags", []):
        updating_fields = {"data": {"ticket": ticket_key}}
        collection.update({"_id": gencfg_group[0]}, {"$set": updating_fields})

    updating_fields = {
        "state": record["state"] + 1,
        "ticket": ticket_key
    }

    logger.debug("Created ticket for  record_id {}: {}".format(record["_id"], ticket_key))
    update_collection_record(record["_id"], updating_fields)


def strange(record):
    logger.debug("STANGE. Record {} came after DONE".format(record["_id"]))


def skip_stages_to_startrek(record, new_fields):
    updating_fields = {
        'state': Status.ABC,
    }
    updating_fields.update(new_fields)
    update_collection_record(record["_id"], updating_fields)


def combine_gencfg_groups(record):
    logger.info("combine_gencfg_groups for record: '{}'".format(record["_id"]))
    abc_dump = get_service_dump_by_name(record["abc_info"], ABC_REMAPPING)
    abc_service_id = abc_dump['service_id']
    abc_slug = abc_dump['slug'] if 'slug' in abc_dump else None
    logger.info("Get abc_service_id: '{}' form abc_info: '{}'".format(
        abc_service_id, record["abc_info"]))

    yp_clients = {}
    for dc in YP_DC_mapping:
        yp_clients[dc] = build_yp_client(dc)

    nanny_session = requests.Session()
    nanny_session.headers['Authorization'] = 'OAuth {}'.format(TOKEN)

    nanny_services = record["nanny_services"]
    nanny_services = set(nanny_services)
    yp_nanny_services, not_yp_nanny_services = get_not_yp_nanny_services(
        nanny_services, nanny_session)
    gencfg_groups = record["gencfg_groups"]

    gencfg_groups, yp_gencfg_groups = combine_gencfg_and_nanny_groups(
        gencfg_groups, nanny_services, nanny_session, yp_clients)

    logger.info("Gencfg groups for record_id {}: {}".format(record["_id"], gencfg_groups))

    not_allowed_groups = []
    unique_gencfg_groups_with_tags = []
    unique_gencfg_groups = mongo_db["unique_gencfg_groups"]
    not_handled_groups = []
    handled_groups = []

    user_groups = []
    gencfg_dump = get_service_dump_by_name(record["yandex_login"], GENCFG_GROUPS)
    if gencfg_dump:
        user_groups = gencfg_dump["groups"]

    stop_reason, guest_groups = get_stop_reason_or_none(gencfg_groups)
    if stop_reason:
        skip_stages_to_startrek(record, {
            "stop_reason": stop_reason,
            "abc_service_id": abc_service_id,
            "slug": abc_slug,
            "guest_groups_with_parents": guest_groups
        })
        return

    for gencfg_group in gencfg_groups:
        if gencfg_group[0] not in user_groups:
            not_allowed_groups.append(gencfg_group[0])
        else:
            try:
                unique_gencfg_groups.insert({"_id": gencfg_group[0]})
                unique_gencfg_groups_with_tags.append(gencfg_group)
                not_handled_groups.append(gencfg_group[0])
            except:
                logger.warn(
                    "Try to insert duplicate gencfg group for record_id {}: {}"
                    .format(record["_id"], gencfg_group[0]))
                handled_groups.append(gencfg_group[0])

    logger.debug("Not allowed groups: {}".format(not_allowed_groups))
    logger.debug("Handled groups: {}".format(handled_groups))

    update_collection_record(record["_id"], {
        "abc_service_id": abc_service_id,
        "slug": abc_slug,
        "not_allowed_groups": not_allowed_groups,
        "yp_nanny_services": [item.id() for item in yp_nanny_services],
        "yp_gencfg_groups": list(yp_gencfg_groups),
        "handled_gencfg_groups": handled_groups,
        "unique_gencfg_groups_with_tags": unique_gencfg_groups_with_tags,
        "state": record["state"] + 1
    })


def change_state_to_done(record):
    logger.info("Change status for record_id {} to DONE".format(record["_id"]))
    update_collection_record(
        record["_id"],
        {"state": record["state"] + 1, "showState": record["state"] + 1},
        True
    )


def handle_record(record):
    if record["state"] < Status.STARTREK:
        update_collection_record(record['_id'],
                                 {'showState': record['state'] + 1})
    record_operations[record["state"]](record)


record_operations = {
    Status.START: combine_gencfg_groups,
    Status.NANNY: estimate_account,
    Status.GENCFG: request_quota_in_abc_service,
    Status.ABC: allocate_quota_and_create_ticket,
    Status.STARTREK: change_state_to_done,
    Status.DONE: strange
}
