# -*- coding: utf-8 -*-

from infra.yp_quota_distributor.lib.common import (
    update_collection_record,
    get_service_dump_by_name
)
from infra.yp_quota_distributor.lib.constants import ABC_REMAPPING
import infra.yp_quota_distributor.lib.startrek_api as startrek_api
import infra.yp_quota_distributor.lib.record_handlers as record_handlers
import infra.yp_quota_distributor.lib.walle_api_wrapper as walle_api_wrapper

from infra.yp_quota_distributor.lib.hosts_quota import (
    calculate_and_format_quota_from_hosts_file
)

from status import Status
import time


problem_resolver = 'glebskvortsov'


def make_error(record, error_msg, updating_fields=None, summonees=None):
    if updating_fields is None:
        updating_fields = {}
    if summonees is None:
        summonees = [problem_resolver]
    DEBUG = record['debug'] if 'debug' in record.keys() else 0
    print(record)
    updating_fields.update({'error_reason': error_msg})
    main_ticket = record['main_ticket']
    startrek_api.add_comment_to_the_ticket(
        main_ticket,
        '**При вводе возникла ошибка:**\n{}'.format(str(error_msg)),
        summonees=summonees
    )
    if 'walle_scenario_ids' in record.keys() and not DEBUG:
        for scenario in record['walle_scenario_ids']:
            walle_api_wrapper.cancel_scenario(scenario)
    update_collection_record(record['_id'], updating_fields=updating_fields, finish_flag=True)


def skip_all_other_stages(record, new_fields):
    updating_fields = {
        'state': Status.ISSUE_QUOTA,
    }
    updating_fields.update(new_fields)
    update_collection_record(record["_id"], updating_fields)


def calculate_quota_and_create_ticket(record):
    DEBUG = record['debug'] if 'debug' in record.keys() else 0
    abc_dump = get_service_dump_by_name(record["abc_info"], ABC_REMAPPING)
    abc_service_id = abc_dump['service_id']

    description, quotas = calculate_and_format_quota_from_hosts_file(
        record['hosts'], record['abc_info'], record['abc_segment'])

    by_dc = quotas.get_total_quota()
    stop = bool(by_dc['total'].gpu_models)
    if stop:
        description.append("**Автоматический ввод хостов с GPU и получения за них квоты пока не поддерживается**")

    ticket_key = startrek_api.create_ticket(
        queue='YPRES',
        summary='Квота в YP для ABC сервиса: {}'.format(record['abc_info']),
        assignee=record['yandex_login'],
        description=description,
        abc_service_id=int(abc_service_id),
        tags=['quota_for_hosts'],
        followers=[record['yandex_login']]
    )['key']
    time.sleep(2)

    update_keys = {
        'main_ticket': ticket_key
    }
    if stop:
        skip_all_other_stages(record, update_keys)
        return

    if DEBUG:
        startrek_api.add_comment_to_the_ticket(ticket_key, 'Вы выбрали режим "только посмотреть". В этом режиме квота не выдается, хосты никуда не вводятся. В тикет печатается только расчет квоты, которую можно получить за эти хосты.')

    update_keys = {
        'main_ticket': ticket_key,
        'hosts_processing': [{
            'fqdn': host['fqdn'],
            'inv': host['inv'],
            'location': host['location']
        } for host in quotas.get_individual_specs()],
        'errors': quotas.get_errors()
    }

    approved_comment = '**FQDN / Инвентарные номера хостов, направленных на ввод**\n%%\n{}\n%%'.format(
        '\n'.join([quota['fqdn'] + '/' + quota['inv'] for quota in quotas.get_individual_specs()])
    )
    errors_comment = '**При расчёте квоты возникли следующие ошибки:**\n%%\n{}\n%%'.format(
        '\n\n'.join([error for error in quotas.get_errors()])
    ) + '\n' + 'Чтобы ввести данные хосты, необходимо исправить данные ошибки, а затем создать новую заявку.'

    # add comment about fails
    if quotas.get_errors():
        update_keys.update({'errors': quotas.get_errors()})
        startrek_api.add_comment_to_the_ticket(ticket_key, errors_comment)

    if not quotas.get_individual_specs():
        record.update(update_keys)
        make_error(record, 'Ни один хост не был найден в боте', update_keys, summonees=[])
        startrek_api.close_ticket(ticket_key, 'won\'tFix')
        return
    # add comment about successes
    if not DEBUG:
        startrek_api.add_comment_to_the_ticket(ticket_key, approved_comment)

    update_keys.update({'account_per_dc': dict()})
    for dc, quota in by_dc.iteritems():
        if dc == 'total':
            continue
        update_keys['account_per_dc'].update(
            {dc: quota.__dict__}
        )

    update_keys.update({'state': record['state'] + 1})
    update_collection_record(record['_id'], update_keys)
    record.update(update_keys)
    return record


def create_walle_scenario(record):
    DEBUG = record['debug'] if 'debug' in record.keys() else 0
    main_ticket = record['main_ticket']
    responsible = record['yandex_login']
    abc_segment = record['abc_segment']
    abc_dump = get_service_dump_by_name(record["abc_info"], ABC_REMAPPING)
    abc_service_id = abc_dump['service_id']
    # prepare for creating multiple scenarios in multiple tickets: one per dc
    hosts_per_dc = dict()
    for host in record['hosts_processing']:
        if host['location'] not in hosts_per_dc.keys():
            hosts_per_dc[host['location']] = []
        hosts_per_dc[host['location']].append(host)

    walle_scenario_ids = []
    if not DEBUG:
        for dc, hosts in hosts_per_dc.items():
            target_project_id = 'yp-iss-{}{}'.format(dc.lower(), '-dev' if abc_segment == 'dev' else '')
            summary = 'Перенос хостов в Wall-e проект {}'.format(target_project_id)
            # create ticket
            description = ('Тип работ:\n'
                           'автоматический ввод/перемещение хостов в wall-e проект\n\n'
                           'Wall-e проект:\n{}\n\n'
                           'Кто сдаёт хосты:\n@{}\n\n'
                           '<{{Список ID серверов:\n'
                           '{}\n'
                           '}}>').format(target_project_id, responsible, '\n'.join(host['inv'] for host in hosts))

            ticket_key = startrek_api.create_ticket(
                queue='RUNTIMECLOUD' if not DEBUG else 'YPRES',
                summary=summary,
                assignee=record['yandex_login'],
                description=description,
                abc_service_id=int(abc_service_id),
                tags=['hosts-comp:move'],
                components=startrek_api.find_components('RUNTIMECLOUD', ['rtc_maintenance', 'hosts'])
            )['key']
            time.sleep(2)

            # link main ticket to this ticket
            startrek_api.add_link(main_ticket, ticket_key)
            # create wall-e scenario, extract scenario id
            response = walle_api_wrapper.create_add_hosts_rtc_scenario(
                name='Automatic hosts add to {} ({})'.format(target_project_id, main_ticket),
                ticket_key=ticket_key,
                responsible=responsible,
                hosts=[int(host['inv']) for host in hosts],
                target_project_id=target_project_id,
                autostart=True
            )
            walle_scenario_ids.append(response['scenario_id'])
    update_keys = {
        'scenarios_in_progress': 1,
        'walle_scenario_ids': walle_scenario_ids,
        'state': record['state'] + 1
    }
    if DEBUG:
        update_keys = {
            'scenarios_in_progress': 0,
            'state': record['state'] + 1,
            "walle_scenarios_finished": 1
        }
    update_collection_record(record['_id'], update_keys)
    record.update(update_keys)
    return record


def check_hosts_consistency(record):
    DEBUG = record['debug'] if 'debug' in record.keys() else 0
    if 'walle_scenarios_finished' not in record.keys():
        return
    main_ticket = record['main_ticket']
    hosts_processing = record['hosts_processing']
    abc_segment = record['abc_segment']

    update_keys = dict()

    scenario_hosts = []
    if DEBUG and 'scenario_hosts' in record.keys():
        scenario_hosts = record['scenario_hosts']
    elif DEBUG and 'scenario_hosts' not in record.keys():
        scenario_hosts = [host['inv'] for host in hosts_processing]
    elif not DEBUG:
        for scenario_id in record['walle_scenario_ids']:
            scenario_hosts.extend(str(host['inv']) for host in walle_api_wrapper.get_scenario(scenario_id, fields=['hosts'])['hosts'])

    if not DEBUG:
        for i in range(len(scenario_hosts)):
            inv = scenario_hosts[i]
            host_walle_info = walle_api_wrapper.get_hosts(invs=[int(inv)])
            if not host_walle_info['result']:
                # пытаемся искать смену инвентарного номера у хоста
                new_inv = walle_api_wrapper.find_hosts_new_inv(int(inv))
                if new_inv:
                    # если нашли - меняем в списке вводимых хостов
                    scenario_hosts[i] = str(new_inv)
                    host_walle_info = walle_api_wrapper.get_hosts(invs=[new_inv])
            if not host_walle_info['result']:
                make_error(record, 'Не удалось получить информацию по хосту "{}" в wall-e'.format(inv))
                return
            host_info = host_walle_info['result'][0]
            target_project_id = 'yp-iss-{}{}'.format(
                host_info['location']['short_datacenter_name'].lower(),
                '-dev' if abc_segment == 'dev' else ''
            )
            if target_project_id != host_info['project']:
                # не все введенные хосты лежат в целевых проектах после того, как сценарий переведен в finished
                make_error(record, 'Некоторые хосты не попали в целевые проекты')
                return

    inv_diff = set([host['inv'] for host in hosts_processing]).difference(set(scenario_hosts))
    if inv_diff:
        # if some hosts changed their inventory numbers (for some reason)
        print(inv_diff)
        print(len(hosts_processing), len(scenario_hosts))
        if len(hosts_processing) == len(scenario_hosts):
            # if there is still the same number of hosts in scenario
            comment, quotas = calculate_and_format_quota_from_hosts_file(
                hosts=scenario_hosts,
                abc_info=record['abc_info'],
                abc_segment=record['abc_segment'],
                # Считаем что фильтрация по проблемным хостам уже сделана и нужно просто пересчитать квоту
                ignore_banned_lists=True,
            )

            comment = '**В связи со сменой инвентарников у некоторых хостов квота будет выдана в соответствии с новым расчётом:**\n{}\n'.format(
                comment
            )
            comment = '{}**Инвентарники, которые были в заявке, но отсутствуют в Wall-e сценарии:**\n%%\n{}\n%%\n'.format(
                comment,
                '\n'.join(inv_diff)
            )
            startrek_api.add_comment_to_the_ticket(
                main_ticket,
                comment,
                summonees=[record['yandex_login']]
            )
            update_keys.update({'account_per_dc': dict()})
            for dc, quota in quotas.get_total_quota().iteritems():
                if dc == 'total':
                    continue
                update_keys['account_per_dc'].update(
                    {dc: quota.__dict__}
                )
            print('recalculated quota')
        else:
            # if number of hosts in not equal to the number of hosts in scenario
            comment = '**Инвентарники в заявке и в Wall-e не совпадают.**\n'
            comment = '{}**Инвентарники, которые были в заявке, но отсутствуют в Wall-e сценарии:**\n%%\n{}\n%%\n'.format(
                comment,
                '\n'.join(inv_diff)
            )
            startrek_api.add_comment_to_the_ticket(
                main_ticket,
                comment,
                summonees=[problem_resolver]
            )
            print('error')
            make_error(record, comment)
            return
    print('adding comment')
    update_keys.update({'state': record['state'] + 1})
    update_collection_record(record['_id'], update_keys)


def issue_quota(record):
    DEBUG = record['debug'] if 'debug' in record.keys() else 0
    if DEBUG:
        startrek_api.close_ticket(record['main_ticket'], 'fixed')
        skip_stage(record)
        return
    segment = record.get('abc_segment', 'default')
    record_handlers.request_quota_in_abc_service(record, full_cores=True, segment=segment)
    startrek_api.add_comment_to_the_ticket(
        record['main_ticket'],
        'Квота выдана в соответствии с последними данными, приведёнными в тикете',
        summonees=[record['yandex_login']]
    )
    startrek_api.close_ticket(record['main_ticket'], 'fixed')
    update_collection_record(record['_id'], {'state': record['state'] + 1})


def change_state_to_done(record):
    update_collection_record(
        record['_id'],
        {'state': record['state'] + 1, 'showState': record['state'] + 1},
        True
    )


def handle_record(record):
    if record['state'] < Status.ISSUE_QUOTA:
        update_collection_record(record['_id'],
                                 {'showState': record['state'] + 1})
    record_operations[record['state']](record)


def pass_func(record):
    pass


def skip_stage(record):
    update_collection_record(record['_id'], {'state': record['state'] + 1})


record_operations = {
    Status.START: calculate_quota_and_create_ticket,    # calculate quota for new request and create ticket in starterk
    Status.QUOTA_AND_TICKET: create_walle_scenario,     # create wall-e scenario
    Status.WALLE_SCENARIO: check_hosts_consistency,     # wall-e scenario complete, check consitency
    Status.CHECK_CONSISTENCY: issue_quota,              # consistency checked, allocate quota
    Status.ISSUE_QUOTA: change_state_to_done,           # quota allocated, change state to done
    Status.DONE: pass_func
}
