# -*- coding: utf-8 -*-

import base64
import logging
import requests
import time
import traceback
import datetime

from collections import defaultdict

from infra.yp_quota_distributor.lib.common import (
    increase_unistat_signal_value,
    logging_wrapper,
    get_collection_by_type,
    get_service_dump_by_name
)
from infra.yp_quota_distributor.lib.constants import (
    TOKEN,
    ABC_API_HOST,
    NANNY_URL,
    GENCFG_API_URL,
    ABC_REMAPPING,
    GENCFG_GROUPS,
    QLOUD_PROJECTS,
    QLOUD_API_URL,
    QLOUD_EXT_API_URL
)
from infra.yp_quota_distributor.lib.log import (
    get_unified_agent_handler,
    get_file_handler
)

from infra.yp_quota_distributor.lib.qloud_api import QloudApi

from infra.yp_quota_distributor.lib.args_parser import parse_args

args = parse_args()

QUOTA_NUMBERS_KEYS = ["cpu", "memory", "hdd", "ssd", "ip4"]


logger = logging.getLogger('infra.yp_quota_distributor')


@logging_wrapper()
def update_abc_services_remapping(abc_remapping):
    abc_session = requests.Session()
    abc_session.headers['Authorization'] = 'OAuth {}'.format(TOKEN)
    services_path = '/api/v4/services/?fields=id,readonly_state,slug,state,owner,name&page_size=1000'

    def build_url():
        return ABC_API_HOST + services_path + '&page=' + str(page_index)

    page_index = 1
    is_next = build_url()
    while is_next:
        response = abc_session.get(is_next)
        if response.ok:
            services = response.json()
            for service in services['results']:
                if service["state"] == "deleted":
                    continue

                is_members_next = ABC_API_HOST + "/api/v4/services/members/?service={}".format(service['id'])
                all_members = []
                capacity_planners = []
                full_members = True

                while is_members_next:
                    res = abc_session.get(is_members_next)
                    if res.ok:
                        res = res.json()
                        is_members_next = res['next']
                        members = res['results']
                        for member in members:
                            if member['role']['code'] == 'capacity_planner':
                                capacity_planners.append(member['person']['login'])

                        members = [item['person']['login'] for item in members]
                        all_members.extend(members)
                    else:
                        full_members = False
                        break
                all_members = list(set(all_members))
                if full_members:
                    if '.' not in service['slug']:
                        data = {
                            'service_id': service['id'],
                            'owner': service['owner']['login'] if service['owner'] else None,
                            'members': all_members,
                            'slug': service['slug'],
                            'capacity_planners': capacity_planners
                        }
                        name = service['name']
                        abc_remapping.update(
                            {
                                "_id": base64.b64encode(name['en'].encode('utf-8'))
                            }, {"$set": data},
                            upsert=True)
                        abc_remapping.update(
                            {
                                "_id": base64.b64encode(name['ru'].encode('utf-8'))
                            }, {"$set": data},
                            upsert=True)
                        abc_remapping.update({"_id": str(service['id'])},
                                             {"$set": data},
                                             upsert=True)
                        abc_remapping.update({"_id": service['slug']},
                                             {"$set": data},
                                             upsert=True)

            is_next = response.json()['next']
            page_index += 1


@logging_wrapper()
def update_gencfg_groups(gencfg_collection):
    groups_owners_path = '/trunk/groups_owners'
    response = requests.get(GENCFG_API_URL + groups_owners_path)
    if response.ok:
        json = response.json()
        gencfg_groups = defaultdict(list)
        for item in json.values():
            for key, value in item.iteritems():
                if key.startswith("abc:"):
                    data = key.split(":")
                    abc_service_name = data[1]
                    abc_dump = get_service_dump_by_name(abc_service_name, ABC_REMAPPING)
                    if abc_dump:
                        for member in abc_dump['members']:
                            gencfg_groups[member].extend(value)
                else:
                    gencfg_groups[key].extend(value)

        for member, groups in gencfg_groups.iteritems():
            gencfg_collection.update({"_id": member}, {'$set': {'groups': groups}}, upsert=True)
        for cursor in gencfg_collection.find():
            if cursor["_id"] not in gencfg_groups:
                gencfg_collection.remove({"_id": cursor["_id"]})


@logging_wrapper()
def update_nanny_services(collection):
    session = requests.Session()
    session.headers['Authorization'] = 'OAuth {}'.format(TOKEN)
    LIMIT = 100
    skip = 0
    services = []

    response_dict = None
    cursor = collection.find_one({'nanny_services': {"$exists": True}})

    while response_dict is None or len(response_dict.get('result')) == LIMIT:
        r = session.get(
            NANNY_URL +
            '/v2/services/?limit={}&skip={}&exclude_runtime_attrs=1'.format(
                LIMIT, skip))
        if r.ok:
            response_dict = r.json()
            for service in response_dict['result']:
                services.append(service['_id'])
            skip += LIMIT

    collection.update({"_id": cursor["_id"]},
                      {'$set': {
                          'nanny_services': services
                      }})


@logging_wrapper()
def update_qloud_projects(qloud_projects_collection):
    qloud_projects = {}
    for main_host, env in zip([QLOUD_API_URL, QLOUD_EXT_API_URL], ["qloud", "qloud-ext"]):
        qloud_client = QloudApi(main_host, TOKEN)
        r = qloud_client.get_all_projects()
        if not r.ok:
            continue
        projects = r.json()
        for project_info in projects:
            project_key = project_info["projectName"]
            r = qloud_client.get_project_access_info(project_key)
            if not r.ok:
                continue
            access_info = r.json()['acls']
            members = set(["glebskvortsov", "atridis"])
            for info in access_info:
                if info['grantedObjectLevel'] == 'Global':
                    if info['type'] == 'USER':
                        members.add(info['user'])
                elif info['grantedObjectLevel'] == 'Project':
                    if info['type'] == 'USER':
                        members.add(info['user'])
                    elif info['type'] == 'DEPARTMENT':
                        info_user = info['user']
                        abc_service_names = [info_user]
                        if '(' in info_user and ')' in info_user:
                            abc_service_names.append(
                                info_user.split('(')[1].split(')')[0])
                        for abc_service_name in abc_service_names:
                            abc_dump = get_service_dump_by_name(abc_service_name, ABC_REMAPPING)
                            if abc_dump:
                                for member in abc_dump['members']:
                                    members.add(member)

            qloud_projects[env + "#" + project_key] = {
                "members": list(members)
            }

    for key, data in qloud_projects.iteritems():
        qloud_projects_collection.update({"_id": key}, {"$set": data}, upsert=True)


def dump_info_into_db():
    logger.addHandler(get_unified_agent_handler())
    logger.addHandler(get_file_handler())
    logger.setLevel(logging.DEBUG)
    logger.info('{} process started'.format(__name__))

    while True:
        try:
            update_gencfg_groups(get_collection_by_type(GENCFG_GROUPS))
            update_abc_services_remapping(get_collection_by_type(ABC_REMAPPING))
            update_qloud_projects(get_collection_by_type(QLOUD_PROJECTS))
        except Exception:
            increase_unistat_signal_value('dump-info-exceptions-number-total', 1)
            logger.exception(
                "Error while dump_info_into_db. Time: {}. Exception {}".format(
                    datetime.datetime.now(), traceback.format_exc()))
        time.sleep(60 * 60 * 4)
