import datetime
import random
import logging
import pymongo
import collections
import requests
import time
import traceback


from infra.yp_quota_distributor.lib.args_parser import parse_args

from infra.yp_quota_distributor.lib.constants import (
    TOKEN,
    NOTIFICATION_TEXT_TEMPLATE,
    ROBOT_GENCFG,
    REMOVE_TIME_FIELD,
    NOTIFICATION_BEFORE_DELETING_TEXT_TEMPLATE,
    GROUPS_WERE_REMOVED_NOTIFICATION_TEMPLATE,
    STARTREK_URL,
    ABC_REMAPPING
)
from infra.yp_quota_distributor.lib.log import (
    get_unified_agent_handler,
    get_file_handler
)
from infra.yp_quota_distributor.lib.mongo import mongo_db
from infra.yp_quota_distributor.lib.common import (
    handle_all_tickets_from_queue_with_filter,
    get_all_dumped_groups,
    logging_wrapper,
    convert_string_to_date,
    convert_date_to_string,
    is_valid_slave,
    get_master_group,
    get_service_dump_by_name
)
from infra.yp_quota_distributor.app.qloud.operations import release_unused_qloud_quota_operation

from infra.yp_quota_distributor.lib import startrek_api
from infra.yp_quota_distributor.lib import gencfg_api
from infra.yp_quota_distributor.lib import nanny_api

QUEUE = "YPRES"
FOURTEEN_DAYS = datetime.timedelta(days=14)
SEVEN_DAYS = datetime.timedelta(days=7)


logger = logging.getLogger('infra.yp_quota_distributor')


def skip_ticket(ticket_key):
    blocked_tickets_collection = mongo_db["blocked_startrek_tickets"]
    is_blocked = blocked_tickets_collection.find_one({"_id": ticket_key})
    is_opened = startrek_api.is_issue_open(startrek_api.get_issue_by_name(ticket_key))
    return (not is_opened) or is_blocked


# Close tickets which ABC creates automatically
@logging_wrapper()
def close_auto_generated_tickets(session, queue):
    def close_handler(session, ticket):
        if ticket.summary.encode("utf-8").startswith("Запрос квот для сервиса"):
            startrek_api.close_ticket(ticket.key, "fixed")

    for status in ['open', 'inProgress']:
        filter_json = {
            'queue': queue,
            'author': ROBOT_GENCFG,
            'status': status
        }
        handle_all_tickets_from_queue_with_filter(session, filter_json, close_handler)


# If group was already handled new ticket with the same group won't be added to handling_startrek_ticket queue.
# That's why robot can't close ticket (it is not in the mapping between groups and tickets) and we have to process it separetly
def get_groups_for_ticket_from_requests_queue(complete_collection, ticket_id):
    cursor = complete_collection.find_one({"ticket": STARTREK_URL + ticket_id, "type": "gencfg"})
    groups = []
    if cursor:
        for group_with_tag in cursor.get("unique_gencfg_groups_with_tags", []):
            groups.append(group_with_tag[0])
    return groups


# Process tickets and corresponding groups from handling_startrek_tickets queue
def get_groups_for_ticket_from_mapping(tickets_collection, ticket_id):
    groups = []
    cursor = tickets_collection.find_one({"_id": ticket_id})
    if cursor:
        for group in cursor["data"]["gencfg_groups"]:
            groups.append(group)

    return groups


# Close ticket if all groups were released
@logging_wrapper()
def close_ticket_with_released_gencfg_groups(session, tickets_collection, complete_collection, queue):
    def close_handler(session, ticket_info, tickets_collection):
        ticket_id = ticket_info.key
        if not tickets_collection.find_one({"_id": ticket_id}):
            return
        corresponding_groups = get_groups_for_ticket_from_requests_queue(complete_collection, ticket_id)
        corresponding_groups.extend(get_groups_for_ticket_from_mapping(tickets_collection, ticket_id))

        groups = get_all_dumped_groups()

        existence_flag = False
        for group in corresponding_groups:
            if group in groups:
                existence_flag = True
                break

        if ticket_info.summary.encode("utf-8").startswith("Квота в YP для ABC сервиса") and not existence_flag:
            startrek_api.close_ticket(ticket_id, "fixed")

    for status in ['open', 'inProgress']:
        filter_json = {
            'queue': queue,
            'author': ROBOT_GENCFG,
            'status': status
        }
        handle_all_tickets_from_queue_with_filter(session, filter_json, close_handler, tickets_collection)


@logging_wrapper()
def add_time_for_unique_gencfg_groups(groups_collection, complete_collection):
    cursor = groups_collection.find({})
    groups = [item for item in cursor]
    cursor = complete_collection.find({"type": "gencfg"})
    records = [item for item in cursor]

    groups_to_request = {}
    for record in records:
        for group_with_tag in record.get("unique_gencfg_groups_with_tags", []):
            groups_to_request[group_with_tag[0]] = record

    result_dict = {}
    for group in groups:
        group_id = group["_id"]
        if group_id in groups_to_request:
            record = groups_to_request[group_id]
            login = record['yandex_login']
            time = record['lastUpdated']
            cur_data = {}
            if group["data"]:
                cur_data.update(group["data"])
            cur_data.update({'time': time, 'login': login, 'ticket': record['ticket'].strip("/").split("/")[-1]})
            result_dict[group_id] = cur_data

    for group, data in result_dict.iteritems():
        groups_collection.update({"_id": group}, {"$set": {"data": data}})


def is_need_to_notify(time):
    return ((datetime.datetime.now() - time) > SEVEN_DAYS)


def send_notification_and_call_owner(ticket, gencfg_groups, logins, days):
    logger.info("Time: {}. Notification for ticket: {}".format(datetime.datetime.now(), ticket))
    startrek_api.add_comment_to_the_ticket(ticket, NOTIFICATION_TEXT_TEMPLATE.format(groups=', '.join(gencfg_groups), days=days), logins)
    startrek_api.set_tags_for_the_ticket(ticket, ["yp_quota", "yp_quota_debt"])


def get_notification_rules(notification_rules_collection):
    rules = notification_rules_collection.find_one({"_id": "rules"})
    rules = rules["data"]
    return rules.get("exception_groups", []), rules.get("special_notification_dates", {})


@logging_wrapper()
def notify_about_group_removing(session,
                                tickets_collection,
                                unique_gencfg_groups_collection,
                                notification_rules_collection):
    exception_groups, special_notification_dates = get_notification_rules(notification_rules_collection)
    existing_groups = get_all_dumped_groups()
    tickets_records = tickets_collection.find()
    inactive_groups = set()

    for record in tickets_records:
        ticket_key = record["_id"]
        if skip_ticket(ticket_key):
            continue
        groups = record["data"]["gencfg_groups"]
        logins = filter(lambda login: login.startswith("abc:") is False and login.startswith("yandex_") is False, record["data"]["logins"])
        time_delta = datetime.datetime.now() - record["data"]["time"]

        for group in groups:
            services = nanny_api.get_services_for_gencfg_group(group)
            inactive_flag = True
            for service in services:
                if nanny_api.is_service_active(service):
                    inactive_flag = False
            if inactive_flag:
                inactive_groups.add(group)

        def handle_group_with_active_service(record):
            data = record.get("data", {})
            if REMOVE_TIME_FIELD in data:
                logger.info("Remove remove_time for group: {}".format(record["_id"]))
                data.pop(REMOVE_TIME_FIELD)
                unique_gencfg_groups_collection.update({"_id": record["_id"]}, {"$set": {"data": data}})

        notified_groups = []
        removed_groups = []
        current_time = datetime.datetime.now()
        remove_date = current_time + SEVEN_DAYS

        def handle_group_with_inactive_service(record):
            data = record.get("data", {})
            logger.info("handle_group_with_inactive_service with group: {} and data: {}".format(record["_id"], data))
            if REMOVE_TIME_FIELD not in data:
                data[REMOVE_TIME_FIELD] = remove_date
                unique_gencfg_groups_collection.update({"_id": record["_id"]}, {"$set": {"data": data}})
                notified_groups.append(record["_id"])
                logger.info("Update remove_time for group: {}".format(record["_id"]))
            elif data[REMOVE_TIME_FIELD] < current_time:
                gencfg_api.remove_group(record["_id"])
                removed_groups.append(record["_id"])

        def skip_group(group):
            return (group not in existing_groups or
                    not get_master_group(group) or
                    group in exception_groups or
                    (group in special_notification_dates
                        and convert_string_to_date(special_notification_dates[group], '%Y-%m-%d') > current_time) or
                    not is_valid_slave(group))

        for group in groups:
            record = unique_gencfg_groups_collection.find_one({"_id": group})
            if not record or skip_group(group):
                continue
            if group in inactive_groups:
                handle_group_with_inactive_service(record)
            else:
                handle_group_with_active_service(record)

        if len(notified_groups) > 0:
            startrek_api.add_comment_to_the_ticket(
                ticket_key,
                NOTIFICATION_BEFORE_DELETING_TEXT_TEMPLATE.format(
                    days=time_delta.days,
                    date=convert_date_to_string(remove_date, '%Y-%m-%d'),
                    groups="\n".join(notified_groups)),
                logins)

        if len(removed_groups) > 0:
            startrek_api.add_comment_to_the_ticket(
                ticket_key,
                GROUPS_WERE_REMOVED_NOTIFICATION_TEMPLATE.format(groups="\n".join(removed_groups)),
                logins)


# REFACTOR THIS FUNCTION
@logging_wrapper()
def create_notification(session, groups_collection,
                        tickets_collection, notification_rules_collection):
    exception_groups, special_notification_dates = get_notification_rules(notification_rules_collection)

    cursor = groups_collection.find({})
    groups = get_all_dumped_groups()
    current_time = datetime.datetime.now()

    def create_empty_state_for_tickets_info():
        return {'gencfg_groups': set(), 'logins': set(), 'time': datetime.datetime.min}

    tickets_info = collections.defaultdict(lambda: create_empty_state_for_tickets_info())

    for group_with_time in cursor:
        if "time" not in group_with_time.get("data", {}):
            continue
        gencfg_group = group_with_time["_id"]

        # TODO: may be use skip_group function
        # skip the group if it is removed or should not be notified
        if (gencfg_group not in groups) or (gencfg_group in exception_groups):
            continue
        # skip the group if set notificationi time less than current
        if (gencfg_group in special_notification_dates
           and (convert_string_to_date(special_notification_dates[gencfg_group], '%Y-%m-%d')) > current_time):
            continue

        updated_time = group_with_time["data"]["time"]
        owner_login = group_with_time["data"]["login"]
        ticket = group_with_time["data"]["ticket"]
        t = current_time - updated_time
        if t > FOURTEEN_DAYS:  # append only if we need to create a nofication
            tickets_info[ticket]['gencfg_groups'].add(gencfg_group)
            try:
                for owner in gencfg_api.get_group_owners(gencfg_group):
                    if owner.startswith("yandex_"):
                        continue
                    if owner.startswith("abc:"):
                        data = owner.split(":")
                        abc_service_name = data[1]
                        abc_dump = get_service_dump_by_name(abc_service_name, ABC_REMAPPING)
                        if abc_dump:
                            for member in abc_dump['members']:
                                tickets_info[ticket]['logins'].add(member)
                    else:
                        tickets_info[ticket]['logins'].add(owner)
                random.shuffle(tickets_info[ticket]['logins'])
                logins_size = len(tickets_info[ticket]['logins'])
                tickets_info[ticket]['logins'] = tickets_info[ticket]['logins'][:min(logins_size, 5)]
                tickets_info[ticket]['logins'].add(owner_login)
            except Exception:
                pass
            tickets_info[ticket]['time'] = max(tickets_info[ticket]['time'], updated_time)

    logger.info("Number of tickets for possible notifying: {}".format(len(tickets_info.keys())))
    for ticket, value in tickets_info.iteritems():
        if skip_ticket(ticket):
            continue

        value["logins"] = list(value["logins"])
        value["gencfg_groups"] = list(value["gencfg_groups"])

        def update_info():
            cursor = tickets_collection.find_one({"_id": ticket})
            cursor['data']['logins'] = value['logins']
            cursor['data']['gencfg_groups'] = value['gencfg_groups']
            cursor['data']['time'] = value['time']
            if is_need_to_notify(cursor["data"]["lastUpdatedTime"]):
                time_delta = datetime.datetime.now() - cursor["data"]["time"]
                send_notification_and_call_owner(ticket,
                                                 value["gencfg_groups"],
                                                 value["logins"],
                                                 time_delta.days)
                cursor['data']['lastUpdatedTime'] = datetime.datetime.now()
            updating_fields = {'data': cursor['data']}
            tickets_collection.update({"_id": ticket}, {"$set": updating_fields})

        last_updated_time = value.get('lastUpdatedTime', datetime.datetime.now())
        try:
            value.update({'lastUpdatedTime': last_updated_time})
            tickets_collection.insert({"_id": ticket, "data": value})
            time_delta = datetime.datetime.now() - value["time"]
            send_notification_and_call_owner(ticket, value["gencfg_groups"], value["logins"], time_delta.days)
        except pymongo.errors.DuplicateKeyError:
            try:
                update_info()
            except Exception:
                logger.exception(traceback.format_exc())
        except Exception:
            logger.exception(traceback.format_exc())


def notify_owners():
    logger.addHandler(get_unified_agent_handler())
    logger.addHandler(get_file_handler())
    logger.setLevel(logging.DEBUG)
    logger.info('{} process started'.format(__name__))
    args = parse_args()
    if args.test:
        # message does not get to logbroker without sleep for some reason
        logger.info("testing environment doesn't make notification")
        time.sleep(2)
        return

    while(True):
        try:
            session = requests.Session()
            session.headers['Authorization'] = 'OAuth {}'.format(TOKEN)

            operations_collection = mongo_db["operations_collection"]
            operations_collection.ensure_index("lockedAt", expireAfterSeconds=12 * 60 * 60)
            operations_collection.insert({"_id": "nofity_owners", "lockedAt": datetime.datetime.now()})
            logger.info('Time: {}. Start notifying'.format(datetime.datetime.now()))

            groups_collection = mongo_db["unique_gencfg_groups"]
            complete_collection = mongo_db["completed_requests"]
            tickets_collection = mongo_db["handling_startrek_tickets"]
            notification_rules_collection = mongo_db["notification_rules"]
            qloud_quota_collection = mongo_db["qloud_quota"]

            release_unused_qloud_quota_operation(complete_collection, qloud_quota_collection)
            close_ticket_with_released_gencfg_groups(session,
                                                     tickets_collection,
                                                     complete_collection,
                                                     QUEUE)
            close_auto_generated_tickets(session, QUEUE)
            add_time_for_unique_gencfg_groups(groups_collection, complete_collection)
            notify_about_group_removing(session, tickets_collection,
                                        groups_collection,
                                        notification_rules_collection)
            create_notification(session, groups_collection, tickets_collection, notification_rules_collection)

            operations_collection.remove({"_id": "nofity_owners"})
            logger.info('Time: {}. Finish notifying'.format(datetime.datetime.now()))
        except pymongo.errors.DuplicateKeyError:
            pass
        except Exception:
            operations_collection.remove({"_id": "nofity_owners"})
            logger.exception("Time {}. Exception while notifying: {}".format(
                datetime.datetime.now(), traceback.format_exc()))
        time.sleep(2 * 60 * 60)
