# -*- coding: utf-8 -*-

import json

from wtforms import StringField, SubmitField, TextAreaField, SelectField, BooleanField
from flask_wtf import FlaskForm
from wtforms.validators import DataRequired

from infra.yp_quota_distributor.lib.constants import (
    STARTREK_URL,
    ADMINS,
    QLOUD_PROJECTS,
    ABC_REMAPPING
)

from infra.yp_quota_distributor.lib.common import get_service_dump_by_name

from infra.yp_quota_distributor.lib.mongo import mongo_db
from infra.yp_quota_distributor.lib import startrek_api
from infra.yp_quota_distributor.lib.qloud_quota import QloudQuotaTable, QloudQuota, QloudBalancerQuota

QLOUD = "qloud"
QLOUD_EXT = "qloud-ext"
QLOUD_ENV_CHOICES = [("https://qloud-ext.yandex-team.ru/", QLOUD_EXT),
                     ("https://qloud.yandex-team.ru/", QLOUD)]

ENV_REMAPPING = {
    "https://qloud-ext.yandex-team.ru/": QLOUD_EXT,
    "https://qloud.yandex-team.ru/": QLOUD
}
BLACK_LIST_TEXT = "This project is blacklisted. Please contact glebskvortsov@ to fix it."


def get_fields_from_form(form):
    abc_info = str(form.abc_service.data).encode("utf-8").strip('/').split('/')[-1]
    tables = json.loads(form.tables.data)
    table_balancer_quota = json.loads(form.table_balancer_quota.data)
    return {
        "qloud_project": form.qloud_project_key.data,
        "qloud_project_url": "{}projects/{}".format(form.qloud_env.data, form.qloud_project_key.data),
        "abc_info": abc_info,
        "comment": form.comment.data.encode("utf-8"),
        "tables": tables,
        "table_balancer_quota": table_balancer_quota,
        "release_qloud_quota": form.release_qloud_quota.data
    }


def _is_project_blacklisted(project_key, blacklist):
    for item in blacklist:
        if item in project_key:
            return True
    return False


class QloudQoutaForm(FlaskForm):
    MAX_PRECISION = 1e5
    qloud_env = SelectField('Choose Qloud env', choices=QLOUD_ENV_CHOICES, validators=[DataRequired()])
    qloud_project_key = StringField('Qloud project (Example: "entitysearch", "zen")')

    show_project_info = SubmitField('Show project quota')
    abc_service = StringField(
        'ABC service id or slug (Example: "gencfg" or "1175")')
    comment = TextAreaField('Comment', default="", render_kw={"rows": 6})
    tables = TextAreaField('Json', description='Volumes json definition', default="")
    table_balancer_quota = TextAreaField('Json', description='Volumes json definition for balancer cpu quota', default="")
    release_qloud_quota = BooleanField('Release qloud quota immediately')
    issue_quota = SubmitField('Issue the quota')

    def __init__(self, owner, *args, **kwargs):
        super(QloudQoutaForm, self).__init__(*args, **kwargs)
        self.owner = owner

    def validate_qloud_project(self):
        qloud_key = ENV_REMAPPING[self.qloud_env.data] + "#" + self.qloud_project_key.data
        qloud_project_dump = get_service_dump_by_name(qloud_key, QLOUD_PROJECTS)
        if not qloud_project_dump:
            self.qloud_project_key.errors.append(
                "Not vaild qloud project: '{}'".format(self.qloud_project_key.data))
            return False

        notification_rules_collection = mongo_db["notification_rules"]
        rules = notification_rules_collection.find_one({"_id": "rules"})
        rules = rules["data"]
        qloud_ext_black_list = rules.get("exception_qloud_ext", [])
        qloud_black_list = rules.get("exception_qloud", [])
        if (ENV_REMAPPING[self.qloud_env.data] == QLOUD_EXT
                and _is_project_blacklisted(self.qloud_project_key.data, qloud_ext_black_list)
                and self.owner not in ADMINS):
            self.qloud_project_key.errors.append(BLACK_LIST_TEXT)
            return False

        if (ENV_REMAPPING[self.qloud_env.data] == QLOUD
                and _is_project_blacklisted(self.qloud_project_key.data, qloud_black_list)
                and self.owner not in ADMINS):
            self.qloud_project_key.errors.append(BLACK_LIST_TEXT)
            return False

        if self.owner not in qloud_project_dump["members"]:
            self.qloud_project_key.errors.append("You cannot choose qloud project: '{}'".format(qloud_key))
            return False
        return True

    def custom_validate_qloud_project(self):
        super(QloudQoutaForm, self).validate()
        self.validate_qloud_project()
        for field in self:
            if field.name != "qloud_project_key":
                field.errors = []
            elif len(field.errors) > 0:
                return False
        return True

    def validate(self):
        fields = get_fields_from_form(self)

        def validate_abc_service(abc_info, owner):
            abc_dump = get_service_dump_by_name(abc_info, ABC_REMAPPING)
            if not abc_dump:
                self.abc_service.errors.append("Not ABC service: '{}'".format(
                    str(abc_info)))
                return False

            if owner not in abc_dump["members"]:
                self.abc_service.errors.append(
                    "You cannot choose ABC service: '{}'".format(
                        str(abc_info)))
                return False
            return True

        def validate_json_tables(tables):
            max_exceed = False
            invalid_cell = False
            negative_number = False
            precision_error = False
            max_cpu_or_memory = False

            def find_table_by_id(id):
                return next(QloudQuotaTable(table) for table in tables if table["id"] == id)

            def validate_cells(table):
                return table.quota.is_value_satisfied(lambda x: True)

            account_per_dc = QloudQuota()
            segments = set([table["segment"] for table in tables])
            for segment in segments:
                in_table = find_table_by_id(segment + "_editable")
                max_table = find_table_by_id(segment + "_current")

                invalid_cell = invalid_cell or not validate_cells(in_table)

                if invalid_cell:  # don't handle NOT float values
                    break

                max_exceed = max_exceed or not (in_table.quota <= max_table.quota)
                max_cpu_or_memory = max_cpu_or_memory or in_table.quota.exceed_max_cpu_or_memory_quota()

                negative_number = negative_number or not in_table.quota.is_value_satisfied(
                    lambda x: x >= 0)
                precision_error = precision_error or not in_table.quota.is_value_satisfied(
                    lambda value: int(value * self.MAX_PRECISION) == value * self.MAX_PRECISION)

                cur_segment_account_per_dc = in_table.quota
                cur_segment_account_per_dc.float_all_values()
                account_per_dc.add(cur_segment_account_per_dc)

            max_cpu_or_memory = max_cpu_or_memory or account_per_dc.exceed_max_cpu_or_memory_quota()

            if invalid_cell:
                self.tables.errors.append("The entered quota should be empty, integer or float (e.g. 10 or 12.5)")
            elif max_exceed:
                self.tables.errors.append("The entered quota must not exceed the maximum")
            elif max_cpu_or_memory:
                self.tables.errors.append("You've exceeded the maximum number of cores/mem available for automatic transfer from Qloud to YP")
            elif negative_number:
                self.tables.errors.append("the entered quota must be non-negative")
            elif precision_error:
                self.tables.errors.append("Use precition 1e-5 or less")
            else:
                return True
            return False

        def validate_json_table_balancer_quota(table_balancer_quota, tables):
            def find_table_by_id(id):
                return next(QloudQuotaTable(table) for table in tables if table["id"] == id)
            account_per_dc = QloudQuota()
            segments = set([table["segment"] for table in tables])
            for segment in segments:
                in_table = find_table_by_id(segment + "_editable")
                cur_segment_account_per_dc = in_table.quota
                cur_segment_account_per_dc.float_all_values()
                account_per_dc.add(cur_segment_account_per_dc)

            assert(len(table_balancer_quota) == 1)
            balancer_quota = QloudBalancerQuota(table_balancer_quota[0]['quota'])
            if not balancer_quota.is_value_satisfied(lambda x: True):
                self.table_balancer_quota.errors.append("The entered quota should be integer")
                return False
            if not balancer_quota.is_value_satisfied(lambda x: x >= 0):
                self.table_balancer_quota.errors.append("The entered quota must be non-negative")
                return False

            balancer_quota.int_all_values()
            balancer_per_dc = balancer_quota.calculate_cpu_per_dc()
            for dc, value in balancer_per_dc.iteritems():
                if value > account_per_dc.quota[dc]['cpu']:
                    self.table_balancer_quota.errors.append("You've requested more CPU for balancers than CPU requested to move from Qloud project to YP")
                    return False
            return True

        valid = super(QloudQoutaForm, self).validate()
        valid = validate_json_tables(fields["tables"]) and valid
        if "table_balancer_quota" in fields and len(fields["table_balancer_quota"]) > 0:
            valid = valid and validate_json_table_balancer_quota(fields["table_balancer_quota"], fields["tables"])  # valid first because table could be invalid
        valid = validate_abc_service(fields["abc_info"], self.owner) and valid
        valid = self.validate_qloud_project() and valid
        for field in self:
            print(field.errors)
        print(valid)
        return valid


class RevertForm(FlaskForm):
    ticket_key = StringField('Ticket key (Example: "YPRES-1")', validators=[DataRequired()])
    revert_quota = SubmitField('Revert quota from this ticket')

    def __init__(self, owner, *args, **kwargs):
        super(RevertForm, self).__init__(*args, **kwargs)
        self.owner = owner

    def validate(self):

        def _validate_ticket_key():
            complete_requests = mongo_db["completed_requests"]
            ticket_key = self.ticket_key.data.encode("utf-8")
            record = complete_requests.find_one({"ticket": STARTREK_URL + ticket_key})

            if not startrek_api.is_issue_exist(ticket_key):
                self.ticket_key.errors.append("Ticket does not exist")
                return False

            if not startrek_api.is_issue_open(ticket_key):
                self.ticket_key.errors.append("Ticket does not open")
                return False

            if startrek_api.get_issue_by_name(ticket_key).tags != [u'qloud_migration']:
                self.ticket_key.errors.append("Ticket has extra tags besides 'qloud_migration'")
                return False

            if not record:
                self.ticket_key.errors.append("There is no database record with this ticket")
                return False

            return True

        valid = super(RevertForm, self).validate()
        valid = _validate_ticket_key() and valid
        return valid
