import copy
import datetime
import traceback
import pandas as pd

from lock_qloud_project import (
    lock_project,
    release_project
)

from infra.yp_quota_distributor.lib.qloud_quota import (
    QloudQuota,
    QloudQuotaTable,
    MAPPING_TO_QLOUD_RESOURCE,
    resources,
)

from infra.yp_quota_distributor.lib.common import (
    pretty_print_pandas_table,
    logging_wrapper,
    logging
)

from infra.yp_quota_distributor.lib.qloud_api import QloudApi
from infra.yp_quota_distributor.lib import startrek_api

from infra.yp_quota_distributor.lib.constants import TOKEN, ROBOT_GENCFG

logger = logging.getLogger(__name__)


@logging_wrapper(True, True)
def _move_quota_from_delta_to_allocated(qloud_quota_collection, project_url, segment, quota):
    record = qloud_quota_collection.find_one({"_id": project_url})
    delta_tables = record["delta_tables"]
    if segment not in delta_tables:
        return
    allocated_tables = record["allocated_tables"]

    if segment not in allocated_tables:
        # create empty tables (delta - delta = 0)
        allocated_tables[segment] = QloudQuotaTable(delta_tables[segment])
        allocated_tables[segment].quota.remove(allocated_tables[segment].quota)
        allocated_tables[segment] = allocated_tables[segment].to_json()

    delta_table = QloudQuotaTable(delta_tables[segment])
    delta_table.quota.remove(quota)
    delta_tables[segment] = delta_table.to_json()

    allocated_table = QloudQuotaTable(allocated_tables[segment])
    allocated_table.quota.add(quota)
    allocated_tables[segment] = allocated_table.to_json()

    result = qloud_quota_collection.update({"_id": project_url}, {
        "$set": {
            "delta_tables": delta_tables,
            "allocated_tables": allocated_tables
        }
    })
    logger.info(
        "Update qloud quota for project_url {} with result {}. Data: delta_tables={}, allocated_tables={}"
        .format(project_url, result, delta_tables, allocated_tables))


@logging_wrapper(True, True)
def _decrease_qloud_quota(project_url, project_key, segment, input_quota, rows):
    qloud_client = QloudApi(project_url, TOKEN)

    full_removed_flag = True
    decreased_quota = copy.deepcopy(input_quota)
    quota = copy.deepcopy(input_quota)
    for key, value in quota.iteritems():
        for resource, num in value.iteritems():
            if resource not in resources:
                continue
            if abs(num - 0) < 1e-9:
                continue
            if resource in MAPPING_TO_QLOUD_RESOURCE:
                r = qloud_client.decrease_quota(
                    project_key, key.upper(), segment,
                    MAPPING_TO_QLOUD_RESOURCE[resource], num)
                print(r.text)
                if r.ok:
                    rows.append([segment, key.upper(), resource, num, "removed"])
                    quota[key][resource] = 0
                else:
                    rows.append([segment, key.upper(), resource, num, "error"])
                    full_removed_flag = False
                    decreased_quota[key][resource] = 0

    return (quota, decreased_quota, full_removed_flag)


def _update_remove_pending_quota(complete_collection, request, quota):
    complete_collection.update(
        {"_id": request["_id"]},
        {"$set": {
            "remove_pending_account_per_dc": quota
        }})


def _get_remove_pending_quota(complete_collection, request):
    if 'remove_pending_account_per_dc' not in request:
        remove_pending_quota = request["segments_account_per_dc"]
        _update_remove_pending_quota(complete_collection, request, remove_pending_quota)
    else:
        remove_pending_quota = request["remove_pending_account_per_dc"]
    return remove_pending_quota


def release_qloud_quote_for_record(complete_collection, qloud_quota_collection, record):
    ticket_key = record["ticket"].split("/")[-1]
    issue = startrek_api.get_issue_by_name(ticket_key)
    tags = issue.tags
    if (startrek_api.is_issue_open(issue)
            or 'qloud_migration_complete' not in tags
            or 'qloud_quota_removed' in tags):
        return

    try:
        last_time = datetime.datetime.now() - datetime.timedelta(days=2)
        if "lastRemoveTryTime" in record:
            last_time = record["lastRemoveTryTime"]
        if datetime.datetime.now() - last_time < datetime.timedelta(days=1):
            return
        complete_collection.update(
            {"_id": record["_id"]},
            {"$set": {
                "lastRemoveTryTime": datetime.datetime.now()
            }},
            upsert=True)
        end_time = datetime.datetime.now() + datetime.timedelta(minutes=20)
        if lock_project(record["qloud_project_url"], ROBOT_GENCFG, end_time):
            remove_pending_quota = _get_remove_pending_quota(complete_collection, record)
            comment = []
            rows = []

            full_removed_flag = True
            new_pending_quota = remove_pending_quota
            for key, value in remove_pending_quota.iteritems():
                quota = QloudQuota()
                quota.set_quota(value)
                new_quota, decreased_quota, cur_full_removed_flag = _decrease_qloud_quota(
                    record["qloud_project_url"], record["qloud_project"],
                    key, quota.quota, rows)
                full_removed_flag = full_removed_flag and cur_full_removed_flag
                new_pending_quota[key] = new_quota
                _update_remove_pending_quota(complete_collection, record, new_pending_quota)
                quota = QloudQuota()
                quota.set_quota(decreased_quota)
                _move_quota_from_delta_to_allocated(qloud_quota_collection,
                                                    record["qloud_project_url"],
                                                    key, quota)

            _update_remove_pending_quota(complete_collection, record, new_pending_quota)
            table = pd.DataFrame(rows, columns=[u"Segment", u"DC", u"Resource", u"Value", u"Status"])
            if rows:
                table = pretty_print_pandas_table(table, args=None)

            if not table.empty:
                def add_color(data):
                    if data == "removed":
                        return "color: green; font-weight:bold"
                    else:
                        return "color: red; font-weight:bold"

                styler = table.style.applymap(add_color, subset=["Status"]).set_table_attributes('border="1"')
                html = styler.render()

                comment.append("<#{}#>".format(html))
                startrek_api.add_comment_to_the_ticket(
                    issue, "\n".join([line.decode("utf-8") for line in comment]),
                    list(set([record["yandex_login"], "glebskvortsov"])))

            if full_removed_flag:
                startrek_api.set_tags_for_the_ticket(issue.key, ["qloud_quota_removed"])
            release_project(record["qloud_project_url"])
    except Exception:
        startrek_api.add_comment_to_the_ticket(
            issue, traceback.format_exc(),
            [record["yandex_login"], "glebskvortsov"])


@logging_wrapper()
def release_unused_qloud_quota_operation(complete_collection, qloud_quota_collection):
    request_cursor = complete_collection.find({"type": "qloud"})
    for request in request_cursor:
        release_qloud_quote_for_record(complete_collection, qloud_quota_collection, request)
