# -*- coding: utf-8 -*-

import logging
import datetime
from status import Status
from operations import release_qloud_quote_for_record

from infra.yp_quota_distributor.lib.common import (
    update_collection_record,
    get_quota_table_by_accounts,
    get_service_dump_by_name,
    get_balancer_quota_table
)

from infra.yp_quota_distributor.lib.record_handlers import (
    request_quota_in_abc_service,
    YP_DC_mapping,
    YP_Distributor
)

from infra.yp_quota_distributor.lib.constants import (
    ABC_URL,
    ABC_RESOURCES_TEMPLATE,
    STARTREK_URL,
    ABC_REMAPPING
)

from infra.yp_quota_distributor.lib import startrek_api
from infra.yp_quota_distributor.lib.mongo import mongo_db

from infra.yp_quota_distributor.lib.qloud_quota import QloudQuota, QloudQuotaTable, QloudBalancerQuota

QLOUD_DAYS = 21


logger = logging.getLogger(__name__)


def strange(record):
    logger.debug("STANGE. Record {} came after DONE".format(record["_id"]))


def change_state_to_done(record):
    logger.info("Change status for record_id {} to DONE".format(record["_id"]))
    update_collection_record(
        record["_id"],
        {"state": record["state"] + 1, "showState": record["state"] + 1},
        True
    )


def summarize_quota_and_allocate_abc_resources(record):
    print("request_quota_in_abc_service")
    print(record)

    account_per_dc = QloudQuota()
    segments_account_per_dc = {}

    for table_dict in record["tables"]:
        table = QloudQuotaTable(table_dict)
        segment_id = table.id
        if segment_id.endswith("_editable"):
            cur_segment_account_per_dc = table.quota
            cur_segment_account_per_dc.float_all_values()
            cur_segment_account_per_dc.calculate_hdd()

            segments_account_per_dc[table.segment] = cur_segment_account_per_dc.quota
            account_per_dc.add(cur_segment_account_per_dc)

    abc_dump = get_service_dump_by_name(record["abc_info"], ABC_REMAPPING)
    abc_service_id = abc_dump['service_id']
    abc_slug = abc_dump['slug'] if 'slug' in abc_dump else None

    error_flag = account_per_dc.exceed_max_cpu_or_memory_quota()
    account_per_dc.calculate_io_hdd()
    account_per_dc.calculate_net_bandwidth()

    if "table_balancer_quota" in record and len(record["table_balancer_quota"]) > 0:
        table_balancer_quota = QloudQuotaTable(record["table_balancer_quota"][0], for_balancer=True)
        table_balancer_quota.quota.int_all_values()
        balancer_io_hdd_per_dc = table_balancer_quota.quota.calculate_io_hdd_per_dc()
        for dc in account_per_dc.quota:
            account_per_dc.quota[dc]["io_hdd"] += balancer_io_hdd_per_dc[dc]

    new_fields = {
        "segments_account_per_dc": segments_account_per_dc,
        "account_per_dc": account_per_dc.quota,
        "abc_slug": abc_slug,
        "abc_service_id": abc_service_id,
    }
    if error_flag:
        new_fields.update({"error_text": "You've exceeded the maximum number of cores/mem available for automatic transfer from Qloud to YP"})

    update_collection_record(record["_id"], new_fields)
    record.update(new_fields)
    if error_flag:
        update_collection_record(record["_id"], {"state": record["state"] + 1})
    else:
        request_quota_in_abc_service(record, full_cores=True)  # increase state entirely


def create_ticket(record):
    print("allocate_quota_and_create_ticket")

    account_per_dc = record["account_per_dc"]
    request_owner = record["yandex_login"]
    qloud_project_url = record["qloud_project_url"]
    qloud_project = record["qloud_project"]
    abc_slug = record["abc_slug"]
    abc_service_id = record["abc_service_id"]
    comment = record["comment"]

    whole_table_html = ""
    for segment, account in record.get("segments_account_per_dc", {}).iteritems():
        table = get_quota_table_by_accounts(account, full_cores=True)
        print(segment)
        print(table)
        del table["Hdd"]
        del table["Ssd"]
        del table["IO_SSD"]
        del table["IO_HDD"]
        del table["NET_BANDWIDTH"]
        whole_table_html += u"<b>Сегмент: {}</b>".format(segment) + "<br>"
        whole_table_html += table.to_html() + "<br>"

    dc_distributors = {}
    for dc, resources in account_per_dc.iteritems():
        dc_distributors[dc] = "abc" if YP_DC_mapping[dc] == YP_Distributor.abc else "abcd"

    description = []
    description.append(u"**Автор:** {}".format(request_owner))
    description.append(u"**Проект Qloud:** {}".format(qloud_project_url))
    description.append(u"**ABC:** {}".format(ABC_URL + "/services/{}".format(abc_slug)))
    description.append(u"====Квота в YP====")
    description.append("<#" + get_quota_table_by_accounts(
        account_per_dc, full_cores=True, distributors=dc_distributors).to_html() + "#>")
    description.append(u"====Квота Qloud====")
    description.append("<#" + whole_table_html + "#>\n")

    if "table_balancer_quota" in record and len(record["table_balancer_quota"]) > 0:
        table_balancer_quota = QloudBalancerQuota(record["table_balancer_quota"][0]["quota"])
        balancer_quota_table_html = get_balancer_quota_table(table_balancer_quota).to_html()
        description.append(u"====Запрошенная балансерная квота====")
        description.append("<#" + balancer_quota_table_html + "#>\n")

    if "error_text" in record:
        description.append(u"**{}**".format(record["error_text"]))
    else:
        description.append(u"(({} YP ресурсы сервиса в ABC))".format(
            ABC_URL + ABC_RESOURCES_TEMPLATE.format(abc_slug)))
        description.append("====Комментарий====")
        description.append(comment)

    assignee = request_owner

    ticket_key = startrek_api.create_ticket(
        "YPRES",
        "Qloud Квота в YP для ABC сервиса: {}".format(abc_slug),
        assignee,
        '\n'.join([line.encode("utf-8") for line in description]),
        int(abc_service_id),
        ["qloud_migration", "qloud_migration_{}".format(qloud_project)],
        [request_owner]
    )["key"]

    ticket_url = STARTREK_URL + ticket_key
    updating_fields = {"ticket": ticket_url}
    update_collection_record(record["_id"], updating_fields)

    startrek_api.start_ticket_progress(ticket_key)

    if "error_text" in record:
        startrek_api.close_ticket(ticket_key, "won'tFix")
    else:
        comment_text = [
            "Вы сделали запрос на перенос квоты проекта Qloud и YP", "\n",
            "Что нужно знать:",
            "- квота в YP выдана автоматически в объеме, указанном в описании тикета",
            "- на переезд из Qloud отводится 2-3 недели",
            "- после завершения миграции закройте, пожалуйста, этот тикет",
            "- квота из Qloud будет вычтена автоматически через какое-то время после закрытия этого тикета",
            "- квота из Qloud будет вычтена из тех ДЦ и сегментов, которые указаны в описании тикета"
        ]
        if record.get("release_qloud_quota", False):
            comment_text = [
                "Вы сделали запрос на автоматический перенос квоты проекта Qloud и YP.",
                "Выбрана функция моментального снижения квоты Qloud. В YP квота выдана в объеме, указанном в описании тикета."
            ]
        abc_folders_url = "{}/services/{}/folders".format(ABC_URL, abc_slug)
        comment_text.append(
            '\n----\n'
            '!!**Внимание!**!!\n'
            'Квота в YP выдается по новой схеме - в ABCD. '
            'Выданную квоту нужно искать в ABC в разделе "Квоты" - ' + abc_folders_url + '\n'
            'Перед использованием квоту необходимо спустить в аккаунт YP. '
            'FAQ про эту процедуру **((https://wiki.yandex-team.ru/resource-model/stepbystepguide/ тут))**.\n'
            'А общая документация **((https://wiki.yandex-team.ru/intranet/abc/quotas/ тут))**'
        )
        startrek_api.add_comment_to_the_ticket(ticket_key, "\n".join(comment_text), [request_owner])

        startrek_api.set_deadline_for_the_tciket(
            ticket_key,
            datetime.datetime.now() + datetime.timedelta(days=QLOUD_DAYS))

        if record.get("release_qloud_quota", False):
            startrek_api.close_ticket(ticket_key, "fixed")
            startrek_api.set_tags_for_the_ticket(ticket_key, ["qloud_migration_complete"])

    updating_fields = {
        "state": record["state"] + 1
    }
    update_collection_record(record["_id"], updating_fields)


def release_qloud_quota(record):
    if record.get("release_qloud_quota", False):
        release_qloud_quote_for_record(mongo_db["requests_queue"],
                                       mongo_db["qloud_quota"], record)
    update_collection_record(
        record["_id"],
        {"state": record["state"] + 1}
    )


def handle_record(record):
    print(record)
    if record['state'] < Status.RELEASE_QLOUD_QUOTA:
        update_collection_record(record['_id'], {'showState': record['state'] + 1})
    record_operations[record["state"]](record)


record_operations = {
    Status.START: summarize_quota_and_allocate_abc_resources,
    Status.ABC_REQUEST: create_ticket,
    Status.CREATE_TICKET: release_qloud_quota,
    Status.RELEASE_QLOUD_QUOTA: change_state_to_done,
    Status.DONE: strange
}
