import requests
from retrying import retry
from constants import TOKEN, ABC_API_URL


class AbcApi(object):
    def __init__(self, abc_api_url, token):
        self.api_url = abc_api_url
        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(token)

    def _get_request(self, path, cgi_params=None):
        return self.session.get(self.api_url + path, params=cgi_params)

    def _post_request(self, path, json_params):
        return self.session.post(self.api_url + path, json=json_params)

    def _patch_request(self, path, json_params):
        return self.session.patch(self.api_url + path, json=json_params)

    def _delete_request(self, path):
        return self.session.delete(self.api_url + path)

    @retry(stop_max_attempt_number=3)
    def get_resources(self, cgi_params):
        response = self._get_request('/resources/consumers/', cgi_params)
        response.raise_for_status()
        return response

    @retry(stop_max_attempt_number=3)
    def change_request_status(self, resource_id, status):
        if status == "deprived":
            response = self._delete_request('/resources/consumers/{}/'.format(resource_id))
        else:
            response = self._patch_request('/resources/consumers/{}/'.format(resource_id), {"state": status})
        response.raise_for_status()

    @retry(stop_max_attempt_number=3)
    def get_service_info(self, abc_slug):
        response = self._get_request('/services/', {'slug' : abc_slug})
        response.raise_for_status()
        return response

    @retry(stop_max_attempt_number=3)
    def request_abc_quota(self, data):
        response = self._post_request("/resources/request/", json_params=data)
        response.raise_for_status()
        return response

    @retry(stop_max_attempt_number=3)
    def get_all_resources_for_service_with_filter(self, service_id, filter_cgi=""):
        fltr = "?supplier=1979&fields=modified_at,id,state,resource,service&service={}".format(service_id) + filter_cgi
        nxt = "/resources/consumers/" + fltr
        result = []
        while nxt:
            response = self._get_request(nxt)
            response.raise_for_status()
            rr = response.json()
            result.extend(rr["results"])
            nxt = rr['next']
        return result


AbcClient = AbcApi(ABC_API_URL, TOKEN)


def get_all_yp_resources(state, requester):
    cgi_parameters = {}
    cgi_parameters['supplier'] = '1979'
    cgi_parameters['state'] = state
    cgi_parameters['requester'] = requester
    return AbcClient.get_resources(cgi_parameters)


def get_all_granted_resources(service_id, register):
    cgi_parameters = "&state=granted&requester={}".format(register)
    return AbcClient.get_all_resources_for_service_with_filter(service_id, cgi_parameters)


def grant_quota_request(resource_id):
    AbcClient.change_request_status(resource_id, "granted")


def deprive_quota_request(resource_id):
    AbcClient.change_request_status(resource_id, "deprived")


def get_service_info(abc_slug):
    return AbcClient.get_service_info(abc_slug)
