import retrying
import requests


class DispenserApi(object):
    def __init__(self, host, cluster='clouds', oauth=None):
        self.url = '{host}/{cluster}/api/v1'.format(
            host=host,
            cluster=cluster
        )
        self.headers = {
            "Content-Type": "application/json",
            "Accept": "application/json",
        }
        if oauth is not None:
            self.headers.update({
                "Authorization": "OAuth {}".format(oauth)
            })
        self.headers.update({
            "Origin": "yp-quota-distributor.n.yandex-team.ru",
            "Access-Control-Allow-Credentials": "true"
        })

    def get_projects(self, request):
        request_url = self.url + '/projects'
        cgi_parameters = {}
        cgi_parameters.update({'leaf': request.get(u'leaf', False)})
        cgi_parameters.update({'member': request.get(u'members', [])})
        cgi_parameters.update({'project': request.get(u'projects', [])})
        cgi_parameters.update({'responsible': request.get(u'responsibles', [])})
        cgi_parameters.update({'showPersons': request.get(u'showPersons', True)})
        return requests.get(request_url, headers=self.headers, params=cgi_parameters)

    def get_project_info(self, request, project):
        request_url = self.url + '/projects/{}'.format(project)
        cgi_parameters = {}
        cgi_parameters.update({'leaf': request.get(u'leaf', False)})
        cgi_parameters.update({'member': request.get(u'members', [])})
        cgi_parameters.update({'project': request.get(u'projects', [])})
        cgi_parameters.update({'responsible': request.get(u'responsibles', [])})
        cgi_parameters.update({'showPersons': request.get(u'showPersons', True)})
        return requests.get(request_url, headers=self.headers, params=cgi_parameters)

    @retrying.retry(stop_max_attempt_number=3)
    def get_project_parents(self, project_key):
        parent_tree = []
        while project_key != "yandex":
            parent_tree.append(project_key)
            r = self.get_project_info({}, project_key)
            r.raise_for_status()
            parent_key = r.json()["parentProjectKey"]
            project_key = parent_key
        return parent_tree

    def update_max_quota_delta(self,
                               req_id,
                               projects_with_fields,
                               segments,
                               service,
                               ticket_key=None,
                               cookie=None,
                               timeout=2):
        request_url = self.url + '/services/{service}/update-max/deltas'.format(service=service)

        updates_array = []
        for project_key_with_fields in projects_with_fields:
            project_key = project_key_with_fields["key"]
            fields = project_key_with_fields["fields"]
            resources = project_key_with_fields["resources"]
            for resource in resources:
                for field in fields:
                    delta = resource['amount']['value']
                    sign = 'POSITIVE' if delta > 0 else 'NEGATIVE'
                    delta = delta if delta > 0 else -delta
                    updates_array.append({
                        field: {
                            'sign': sign,
                            'amount': {
                                "value": delta,
                                "unit": resource["amount"]["unit"]
                            }
                        },
                        'resourceKey': resource['resource_key'],
                        'projectKey': project_key,
                        'segmentKeys': segments
                    })
        data = {
            'updates': updates_array,
        }
        if ticket_key:
            data.update({"ticketKey": ticket_key})
        headers = dict(self.headers)
        if cookie:
            headers.update({"Cookie": cookie})
        r = None
        timeout_flag = False
        try:
            r = requests.post(request_url,
                              json=data,
                              headers=headers,
                              params={"reqId": req_id},
                              timeout=timeout)
        except requests.exceptions.Timeout:
            timeout_flag = True
        return r, timeout_flag
