# -*- coding: utf-8 -*-

from flask_wtf import FlaskForm
from wtforms import (
    SelectField,
    FloatField
)
from wtforms.validators import DataRequired
from infra.yp_quota_distributor.lib.constants import (
    SEGMENTS_CHOICES,
    DC_CHOICES
)


class YPResourceForm(FlaskForm):
    MAX_VALUE = 1e9
    MAX_PRECISION = 1e5
    segment = SelectField('Choose segment', choices=SEGMENTS_CHOICES, validators=[DataRequired()])
    dc = SelectField('Choose DC', choices=DC_CHOICES, validators=[DataRequired()])
    cpu = FloatField('CPU in CORES', default=0.0)
    memory = FloatField('MEMORY (RAM) in  GB', default=0.0)
    ssd = FloatField('SSD in TB', default=0.0)
    hdd = FloatField('HDD in TB', default=0.0)

    def _validate_float_field(self, field):
        if field.data > self.MAX_VALUE:
            field.errors.append("Quota cannot exceed {}".format(self.MAX_VALUE))
            return False
        if int(field.data * self.MAX_PRECISION) != field.data * self.MAX_PRECISION:
            field.errors.append("Use precition 1e-5 or less")
            return False
        return True

    def validate(self):
        valid = super(YPResourceForm, self).validate()

        def validate_values():
            if self.cpu.data == 0 and self.memory.data == 0 and self.ssd.data == 0 and self.hdd.data == 0:
                self.cpu.errors.append("All quota cannot be empty. Please fill in any field.")
                self.memory.errors.append("All quota cannot be empty. Please fill in any field.")
                self.hdd.errors.append("All quota cannot be empty. Please fill in any field.")
                self.ssd.errors.append("All quota cannot be empty. Please fill in any field.")
                return False

            return True

        valid = valid and validate_values()
        valid = valid and self._validate_float_field(self.cpu)
        valid = valid and self._validate_float_field(self.memory)
        valid = valid and self._validate_float_field(self.hdd)
        valid = valid and self._validate_float_field(self.ssd)
        return valid
