import requests
from retrying import retry
from constants import (
    TOKEN,
    GENCFG_API_URL,
    ROBOT_GENCFG,
    ABC_REMAPPING
)

from common import (
    get_service_dump_by_name
)

from sandbox_api import SandboxExecutor


class GencfgClient(object):
    def __init__(self, gencfg_api_url, token):
        self._init_sandbox(token)
        self.api_url = gencfg_api_url
        self.session = requests.Session()
        self.session.headers['Authroization'] = 'OAuth {}'.format(token)

    def _init_sandbox(self, token):
        self.sandbox = SandboxExecutor(default_owner='GENCFG', token=token)

    def get_request(self, path, cgi_params=None):
        if cgi_params is None:
            cgi_params = {}
        return self.session.get(self.api_url + path, params=cgi_params)

    def wait_sandbox_task(self, custom_fields, description, task_type):
        return self.sandbox.wait_from_params(
            custom_fields, description=description, task_type=task_type
        )


gencfg_client = GencfgClient(GENCFG_API_URL, TOKEN)


def remove_group(group_name, request_author=ROBOT_GENCFG, message='YP_QUOTA_DISTRIBUTOR', dry_run=False):
    """
    Remove specified group from Gencfg. Use this handle with caution.
    :param group_name: type str
    :param request_author: Login of person who asked to remove service (without @)
    :param dry_run: Run sandbox task in dry run mode (without actual commit)
    :param message: Optional text string with any suitable info about operation
    :return: committed svn revision
    :type request_author: str
    :type dry_run: bool
    :rtype: int
    """
    description = 'Removing group(s) {}. Details: {}'.format(group_name, message)
    custom_fields = {
        'group_name': group_name,
        'commit_message': description,
        'dry_run': dry_run,
    }
    if request_author:
        custom_fields['author'] = request_author
    task = gencfg_client.wait_sandbox_task(custom_fields, description, task_type='REMOVE_GROUP')
    gencfg_commit = task.context['commit']
    return gencfg_commit


@retry(stop_max_attempt_number=3)
def get_group_owners(group):
    full_logins = []

    r = gencfg_client.get_request("/trunk/groups/{}/card".format(group))
    r.raise_for_status()
    owners = r.json()["owners"]
    for owner in owners:
        if owner.startswith("abc:"):
            abc_service = owner[4:]
            abc_dump = get_service_dump_by_name(abc_service, ABC_REMAPPING)
            if abc_dump:
                full_logins.extend(abc_dump["members"])
        else:
            full_logins.append(owner)

    return full_logins


@retry(stop_max_attempt_number=3)
def get_cpumodels():
    r = gencfg_client.get_request("/cpumodels")
    r.raise_for_status()
    return r.json()


def get_cpu_cores(models, input_cpu_model, is_bot_model, use_power=True):
    cpu_model = models["botmodel_to_model"][input_cpu_model] if is_bot_model else input_cpu_model
    cores = models["models"][cpu_model]["ncpu"]
    if use_power:
        cores *= (float(models["models"][cpu_model]["power"]) / models["models"][cpu_model]["ncpu"]) / \
                 (float(models["models"]["E5-2650v2"]["power"]) / models["models"]["E5-2650v2"]["ncpu"])
    return cores
