# -*- coding: utf-8 -*-

from collections import defaultdict

from infra.yp_quota_distributor.lib.mongo import mongo_db
from infra.yp.account_estimation import Account

from infra.yp.account_estimation import ceil_with_precision


dcs = ['sas', 'man', 'vla', 'iva', 'myt']
resources = ['cpu', 'memory', 'hdd', 'ssd', 'io_ssd', 'io_hdd', 'net_bandwidth']
qloud_resource_mapping = {
    'CPU': 'cpu',
    'MEMORY_GB': 'memory'
}
MAPPING_TO_QLOUD_RESOURCE = {
    'cpu': 'CPU',
    'memory': 'MEMORY_GB'
}


def default_quota():
    return {resource: "" for resource in resources}


def default_account():
    return Account.empty().__dict__


def zero_qloud_quota():
    return {dc: 0.0 for dc in dcs}


def parse_qloud_quota(value):
    quota = value['quota']

    quota_by_segments = defaultdict(list)
    for item in quota:
        if item['location'].lower() in dcs:
            quota_by_segments[item['hardwareSegment']].append(item)

    result_value = []
    for segment, quota_data in quota_by_segments.iteritems():
        quota_by_resources = defaultdict(zero_qloud_quota)
        for item in quota_data:
            if item['resource'] in qloud_resource_mapping:
                quota_by_resources[qloud_resource_mapping[
                    item['resource']]].update(
                        {item['location'].lower(): item['quota']})

        cur_quota = []
        from pprint import pprint
        pprint(quota_by_resources)
        for resource, item in quota_by_resources.iteritems():
            if resource in resources:
                item.update({'resource': resource})
                cur_quota.append(item)

        result_value.append({
            'id': segment,
            'caption': u'Current available quota',
            'quota': cur_quota,
            'segment': segment
        })

    return result_value


class QloudQuota:
    QLOUD_RESOURCES = ["cpu", "memory"]

    def __init__(self, quota=None):
        if quota is None:
            quota = []
        self.quota = defaultdict(default_account)
        for item in quota:
            resource = item["resource"]
            if resource not in resources:
                continue
            for key, value in item.iteritems():
                if key != "resource":
                    self.quota[key][resource] = value

    def set_quota(self, quota):
        self.quota = quota

    def is_value_satisfied(self, consumer):
        for dc, data in self.quota.iteritems():
            for resource, value in data.iteritems():
                if resource not in resources:
                    continue
                if value != "":
                    try:
                        if not consumer(float(value)):
                            return False
                    except Exception:
                        return False
        return True

    def __le__(self, other):
        for dc in dcs:
            for resource, value in self.quota[dc].iteritems():
                if resource not in resources:
                    continue
                if value != "" and float(value) > float(other.quota[dc].get(resource, 0)):
                    print("compare {} vs {}".format(value, float(other.quota[dc].get(resource, 0))))
                    return False
        return True

    def update(self, new_value):
        for dc in self.quota:
            self.quota[dc].update(new_value)

    # Table format
    def to_json(self):
        res = []
        for resource in self.QLOUD_RESOURCES:
            item = {dc: self.quota[dc][resource] for dc in dcs}
            item.update({"resource": resource})
            res.append(item)
        return res

    def float_all_values(self):
        for dc in self.quota:
            for resource in self.quota[dc]:
                if resource in resources:
                    try:
                        self.quota[dc][resource] = ceil_with_precision(float(self.quota[dc][resource]), 4)
                    except Exception:
                        self.quota[dc][resource] = 0.0

    def calculate_hdd(self):
        for dc in self.quota:
            self.quota[dc]["hdd"] = round(self.quota[dc]["cpu"] * 3.0 / 80.0, 4)

    def add(self, other):
        for dc in dcs:
            for resource in self.quota[dc]:
                if resource in resources:
                    self.quota[dc][resource] += other.quota[dc].get(resource, 0)
                    self.quota[dc][resource] = ceil_with_precision(float(self.quota[dc][resource]), 4)

    def remove(self, other):
        for dc in dcs:
            for resource in self.quota[dc]:
                if resource in resources:
                    self.quota[dc][resource] -= other.quota[dc].get(resource, 0)
                    self.quota[dc][resource] = ceil_with_precision(float(self.quota[dc][resource]), 4)

    def exceed_max_cpu_or_memory_quota(self):
        for dc in dcs:
            cpu = self.quota[dc].get("cpu", 0)
            memory = self.quota[dc].get("memory", 0)
            if (cpu != "" and float(cpu) >= 200) or (memory != "" and float(memory) >= 800):
                return True
        return False

    def calculate_io_hdd(self):
        MIN_IO_HDD = 15
        for dc in dcs:
            self.quota[dc]["io_hdd"] = max(2 * self.quota[dc]["cpu"], MIN_IO_HDD) if self.quota[dc]["cpu"] > 0 else 0

    def calculate_net_bandwidth(self):
        MIN_NET_BANDWIDTH = 10
        for dc in dcs:
            self.quota[dc]["net_bandwidth"] = max(7 * self.quota[dc]["cpu"], MIN_NET_BANDWIDTH) if self.quota[dc]["cpu"] > 0 else 0


class QloudBalancerQuota:
    DISPLAY_POSITION = {
        'nano': 0,
        'micro': 1,
        'small': 2,
        'medium': 3
    }

    BALANCER_MODE_TO_CPU = {
        'nano': 0.5,
        'micro': 1.,
        'small': 2.,
        'medium': 4.
    }

    BALANCER_MODE_TO_IO_HDD = {
        'nano': 15,
        'micro': 15,
        'small': 20,
        'medium': 30
    }

    def __init__(self, quota=None):
        if quota is None:
            quota = []
        self.nano = {dc: 0 for dc in dcs}
        self.micro = {dc: 0 for dc in dcs}
        self.small = {dc: 0 for dc in dcs}
        self.medium = {dc: 0 for dc in dcs}
        for item in quota:
            balancer_mode = item["balancer_mode"]
            for key, value in item.iteritems():
                if key != "balancer_mode":
                    if balancer_mode == 'nano':
                        self.nano[key] = value
                    elif balancer_mode == 'micro':
                        self.micro[key] = value
                    elif balancer_mode == 'small':
                        self.small[key] = value
                    elif balancer_mode == 'medium':
                        self.medium[key] = value

    def to_json(self):
        res = []
        for balancer_mode, per_dc_value in self.__dict__.iteritems():
            item = {"balancer_mode": balancer_mode}
            for dc, value in per_dc_value.iteritems():
                item[dc] = value
            res.append(item)
        res = sorted(res, key=lambda x: self.DISPLAY_POSITION[x["balancer_mode"]])
        return res

    def int_all_values(self):
        for dc in dcs:
            try:
                self.nano[dc] = int(self.nano[dc])
            except Exception:
                self.nano[dc] = 0
            try:
                self.micro[dc] = int(self.micro[dc])
            except Exception:
                self.micro[dc] = 0
            try:
                self.small[dc] = int(self.small[dc])
            except Exception:
                self.small[dc] = 0
            try:
                self.medium[dc] = int(self.medium[dc])
            except Exception:
                self.medium[dc] = 0

    def is_value_satisfied(self, consumer):
        for dc in dcs:
            values = [self.nano[dc], self.micro[dc], self.small[dc], self.medium[dc]]
            for value in values:
                if value != "":
                    try:
                        if not consumer(int(value)):
                            return False
                    except Exception:
                        return False

        return True

    def calculate_cpu_per_dc(self):
        res = {dc: 0.0 for dc in dcs}
        for dc in dcs:
            res[dc] += self.nano[dc] * self.BALANCER_MODE_TO_CPU['nano']
            res[dc] += self.micro[dc] * self.BALANCER_MODE_TO_CPU['micro']
            res[dc] += self.small[dc] * self.BALANCER_MODE_TO_CPU['small']
            res[dc] += self.medium[dc] * self.BALANCER_MODE_TO_CPU['medium']
        return res

    def calculate_io_hdd_per_dc(self):
        res = {dc: 0 for dc in dcs}
        for dc in dcs:
            res[dc] += self.nano[dc] * self.BALANCER_MODE_TO_IO_HDD['nano']
            res[dc] += self.micro[dc] * self.BALANCER_MODE_TO_IO_HDD['micro']
            res[dc] += self.small[dc] * self.BALANCER_MODE_TO_IO_HDD['small']
            res[dc] += self.medium[dc] * self.BALANCER_MODE_TO_IO_HDD['medium']
        return res


class QloudQuotaTable:
    def __init__(self, dictionary, for_balancer=False):
        for k, v in dictionary.items():
            setattr(self, k, v)

        if for_balancer:
            if "quota" in self.__dict__.keys():
                self.quota = QloudBalancerQuota(self.quota)
            else:
                self.quota = QloudBalancerQuota()
        else:
            self.quota = QloudQuota(self.quota)

    def to_json(self):
        res = self.__dict__
        res["quota"] = self.quota.to_json()
        return res


def get_free_qloud_quota(qloud_project_url, table):
    def get_table_for_segment(segment, key):
        qloud_quota_collection = mongo_db["qloud_quota"]
        cursor = qloud_quota_collection.find_one({"_id": qloud_project_url})
        if cursor and key in cursor and segment in cursor[key]:
            return cursor[key][segment]
        return None

    def handle_delta(key):
        additional_quota = get_table_for_segment(table.segment, key)
        if additional_quota:
            table.quota.remove(QloudQuotaTable(additional_quota).quota)
    handle_delta("delta_tables")

    return table
