import logging
from collections import defaultdict
from abc_api import (
    AbcClient,
    get_all_granted_resources
)
from abcd_api import AbcdClient

from common import (
    update_collection_record,
    reject_duplicates_for_service
)

from constants import (
    ROBOT_GENCFG
)


class YP_Distributor:
    abc = 1
    abcd = 2


YP_DC_mapping = {
    "vla": YP_Distributor.abcd,
    "man": YP_Distributor.abcd,
    "sas": YP_Distributor.abcd,
    "myt": YP_Distributor.abcd,
    "iva": YP_Distributor.abcd,
}

d_unit_mappings = {
    "cpu": {
        "type": "cpu_capacity",
        "multiplier": 1000.0,
        "units": "millicores",
    },
    "memory": {
        "type": "memory_limit",
        "multiplier": 1024.0,
        "units": "mebibytes",
    },
    "ssd": {
        "type": "ssd_disk_capacity",
        "multiplier": 1024.0 * 1024.0,
        "units": "mebibytes",
    },
    "hdd": {
        "type": "hdd_disk_capacity",
        "multiplier": 1024.0 * 1024.0,
        "units": "mebibytes",
    },
    "io_ssd": {
        "type": "ssd_disk_bandwidth",
        "multiplier": 1024.0,
        "units": "kibibytesPerSecond",
    },
    "io_hdd": {
        "type": "hdd_disk_bandwidth",
        "multiplier": 1024.0,
        "units": "kibibytesPerSecond",
    },
    "net_bandwidth": {
        "type": "network_bandwidth",
        "multiplier": 1024.0,
        "units": "kibibytesPerSecond",
    },
}


logger = logging.getLogger(__name__)


def prepare_abc_request_data(data, record, dc, res_values, full_cores):

    yp_segment = record.get("abc_segment", "default")
    groups_by_dc = record.get("groups_by_dc", defaultdict(list))

    if "io_ssd" not in res_values:
        res_values["io_ssd"] = 0
    if "io_hdd" not in res_values:
        res_values["io_hdd"] = 0
    if "net_bandwidth" not in res_values:
        res_values["net_bandwidth"] = 0
    quota_data = {
        "comment": "yp quota",
        "location": dc.upper(),
        "segment": yp_segment,
        "cpu": res_values["cpu"] if full_cores else res_values["cpu"] / 1000.0,
        "memory": res_values["memory"],
        "ssd": res_values["ssd"],
        "hdd": res_values["hdd"],
        "io_hdd": res_values["io_hdd"] if full_cores else round(res_values["io_hdd"] / 1000.0),
        "io_ssd": res_values["io_ssd"] if full_cores else round(res_values["io_ssd"] / 1000.0),
        "net_bandwidth": res_values["net_bandwidth"] if full_cores else round(res_values["net_bandwidth"] / 1000.0),
        "gencfg-groups": ','.join(groups_by_dc[dc.lower()][:10])
    }
    data['data'] = quota_data

    logger.debug("ABC data for record_id {}: {}".format(record["_id"], data))


def fill_abcd_transfer_data(data, dc, res_values, provider_id, source_folder_uid, dest_folder_uid, full_cores=False,
                            segment='default'):

    dc = dc.lower()

    for res_name, res_value in res_values.items():

        if res_name not in d_unit_mappings:
            continue
        if not res_value:
            continue
        amount = res_value if full_cores or res_name not in ['cpu', 'io_ssd', 'io_hdd', 'net_bandwidth'] else res_value / 1000.0
        amount = int(round(amount * d_unit_mappings[res_name]['multiplier']))
        new_data = {
            'sourceFolderId': source_folder_uid,
            'destinationFolderId': dest_folder_uid,
            'resource': {
                'providerId': provider_id,
                'externalResourceId': {
                    'typeKey': d_unit_mappings[res_name]['type'],
                    'segmentation': [
                        {
                            'segmentationKey': 'cluster',
                            'segmentKey': dc,

                        },
                        {
                            'segmentationKey': 'node_segment',
                            'segmentKey': segment,
                        },
                    ]
                }
            },
            'amount': amount,
            'amountUnitKey': d_unit_mappings[res_name]['units'],
        }
        data['transfers'].append(new_data)


def request_quota_in_abc_service(record, full_cores=False, segment='default'):

    logger.info("request_quota_in_abc_service for record: '{}'".format(record["_id"]))

    abc_service_id = int(record["abc_service_id"])
    account_per_dc = record["account_per_dc"]
    abc_data = {
        'resource_type': 96,  # YP quota in prod
        'service': abc_service_id
    }

    abcd_data = {
        "transfers": [],
        "comment": "YP quota migration",
    }
    abcd_yp_provider_id = AbcdClient.get_provider_id("YP")
    abcd_source_folder_id = AbcdClient.get_default_folder_id("1979", folder_type="PROVIDER_RESERVE")
    abcd_dest_folder_id = AbcdClient.get_default_folder_id(abc_service_id)
    abcd_dc = []

    dc_success = {}
    for dc in YP_DC_mapping:
        dc_success[dc] = False
    if "dc_abc_request_successed" in record:
        dc_success = record["dc_abc_request_successed"]

    for key, value in account_per_dc.iteritems():
        if all(v == 0 or v == {} for v in value.values()):
            dc_success[key] = True
            continue
        if key.lower() not in YP_DC_mapping:
            logger.error("Unsupported DC '{}' for record_id '{}'".format(key, record["_id"]))
        if dc_success[key]:
            continue
        if YP_DC_mapping[key.lower()] == YP_Distributor.abcd:
            fill_abcd_transfer_data(abcd_data, key, value, abcd_yp_provider_id,
                                    abcd_source_folder_id, abcd_dest_folder_id, full_cores, segment)
            abcd_dc.append(key)
        else:
            prepare_abc_request_data(abc_data, record, key, value, full_cores)
            abc_request_response = AbcClient.request_abc_quota(abc_data)
            if abc_request_response.ok:
                dc_success[key] = True

    # ABCD move bulk request
    if len(abcd_data["transfers"]) > 0:
        logger.info("abcd data {}".format(abcd_data))
        abcd_success = AbcdClient.request_d_quota(abcd_data)
        if abcd_success:
            for dc in abcd_dc:
                dc_success[dc] = True

    reject_duplicates_for_service(
        abc_service_id,
        get_all_granted_resources(service_id=abc_service_id, register=ROBOT_GENCFG)
    )
    state_delta = True
    for key, value in account_per_dc.iteritems():
        state_delta = state_delta and dc_success[key]
    updating_fields = {
        "problem": "ABC replied 500 error, retrying" if not state_delta else "",
        "state": record["state"] + state_delta,
        "dc_abc_request_successed": dc_success
    }
    update_collection_record(record["_id"], updating_fields)
