import time
from six import iteritems
import logging

from sandbox.common.rest import Client
from sandbox.common.proxy import OAuth
import sandbox.common.types.task as ctt

sandbox_logger = logging.getLogger('SandboxClient')


class SandboxApi(object):
    def __init__(self, token=None):
        self.sandbox_client = Client(auth=OAuth(token))

    def add_task(self, task_type, task_owner, task_params, description='', notifications=None):
        assert task_type and task_owner

        custom_fields = [{'name': k, 'value': v} for k, v in iteritems(task_params)]

        sandbox_task_info = self.sandbox_client.task(
            type=task_type,
            description=description,
            owner=task_owner,
            custom_fields=custom_fields,
            notifications=notifications
        )
        return SandboxTask.from_task_info(self.sandbox_client, sandbox_task_info)


class SandboxExecutor(object):
    def __init__(self, default_owner, token, default_notifications=None):
        self.sandbox_api = SandboxApi(token)
        self.default_owner = default_owner
        self.default_notifications = default_notifications

    def add_task(self, params, description=None, task_type=None, owner=None, notifications=None):
        task_type = task_type or self.default_task_type
        owner = owner or self.default_owner
        notifications = notifications or self.default_notifications

        return self.sandbox_api.add_task(task_type=task_type, task_owner=owner, task_params=params, description=description, notifications=notifications)

    def run_from_params(self, *args, **kwargs):
        """
        :rtype: SandboxTask
        """
        task = self.add_task(*args, **kwargs)
        task.start()
        return task

    def wait_from_params(self, *args, **kwargs):
        """
        :rtype: SandboxTask
        """
        task = self.run_from_params(*args, **kwargs)
        task.wait()
        return task


# Provides not more than fixed % delay overhead
# and generates O(log T) requests
def default_delay_generator(N):
    delay = float(N)
    multiplier = 1.05
    while True:
        delay *= multiplier
        yield int(delay)


class SandboxTask(object):
    FINAL_STATES = ctt.Status.Group.BREAK + ctt.Status.Group.FINISH

    def __init__(self, sandbox, task_id):
        """
        :type sandbox: sandbox.common.rest.Client
        :type task_id: int

        """
        self.sandbox_client = sandbox
        self.id = task_id
        self._info = {}

    @classmethod
    def from_task_info(cls, sandbox, task_info):
        """
        :type sandbox: sandbox.common.rest.Client
        :type task_info: dict
        :rtype: SandboxTask
        """
        task = cls(sandbox, task_info['id'])
        task._info = task_info
        return task

    def update_info(self):
        self._info = self.sandbox_client.task[self.id].read()

    def __getattribute__(self, name):
        if name in {'type', 'owner', 'author', 'description'}:
            if self._info.get(name, None) is None:
                self.update_info()
            return self._info[name]
        else:
            return object.__getattribute__(self, name)

    @property
    def status(self):
        if self._info.get('status', None) is None or self._info['status'] not in self.FINAL_STATES:
            self.update_info()
        return self._info['status']

    @property
    def context(self):
        return self.sandbox_client.task[self.id].context.read()

    def start(self):
        self.sandbox_client.batch.tasks.start.update([self.id])

    def wait(self, max_restarts_on_break=3, delay_generator=default_delay_generator(3)):
        sandbox_logger.info('Waiting for task {} to complete...'.format(self.id))
        while True:
            status = self.status
            if status in ctt.Status.Group.SUCCEED:
                sandbox_logger.info('Task %s completed successfully', self.id)
                return self
            elif status in self.FINAL_STATES:
                raise Exception('Task {} finished unsuccessfully with final status {}'.format(self.id, self.status))
            else:
                next_sleep = delay_generator.next()
                sandbox_logger.debug('Waiting task to change state from %s. Next sleep %d', status, next_sleep)
                time.sleep(next_sleep)
