import retrying

from constants import (
    TOKEN,
    STARTREK_API_BASE_URL
)
from startrek_client import Startrek

# https://github.yandex-team.ru/devtools/startrek-python-client
# https://wiki.yandex-team.ru/tracker/api

Client = Startrek(useragent='python', token=TOKEN, base_url=STARTREK_API_BASE_URL)


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def get_issue_by_name(issue):
    if isinstance(issue, (str, unicode,)):
        return Client.issues[issue]
    return issue


# May be use **kwargs
def create_ticket(queue, summary, assignee, description, abc_service_id, tags=None, followers=None, components=None):
    if tags is None:
        tags = []
    if followers is None:
        followers = []
    if components is None:
        components = []
    return Client.issues.create(
        queue=queue,
        summary=summary,
        assignee=assignee,
        description=description,
        tags=tags,
        followers=followers,
        components=components,
        abcService=abc_service_id
    )


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def close_ticket(issue, resolution, comment=None):
    issue = Client.issues[issue]
    transition = issue.transitions['close']
    if comment:
        transition.execute(resolution=resolution, comment=comment)
    else:
        transition.execute(resolution=resolution)


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def reopen_ticket(issue):
    issue = get_issue_by_name(issue)
    issue.transitions['reopen'].execute()


@retrying.retry(stop_max_attempt_number=4, wait_fixed=4000)
def start_ticket_progress(issue):
    issue = get_issue_by_name(issue)
    issue.transitions['start_progress'].execute()


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def add_comment_to_the_ticket(issue, comment, summonees=None):
    if summonees is None:
        summonees = []
    issue = get_issue_by_name(issue)
    comment = issue.comments.create(text=comment, summonees=summonees)


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def set_tags_for_the_ticket(issue, tags):
    issue = get_issue_by_name(issue)
    old_tags = set(issue.tags)
    if set(tags) - old_tags:
        issue.update(tags={'add': tags})


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def set_deadline_for_the_tciket(issue, deadline_time):
    issue = get_issue_by_name(issue)
    issue.update(deadline=deadline_time.strftime("%Y-%m-%d"))


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def get_all_issues_with_filter(filter_json):
    return list(Client.issues.find(
        filter=filter_json,
        scrollType='sorted'
    ))


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def get_issue_author(issue):
    issue = get_issue_by_name(issue)
    return issue.createdBy.id


@retrying.retry(stop_max_attempt_number=3, wait_fixed=4000)
def is_issue_open(issue_key):
    issue = get_issue_by_name(issue_key)
    return issue.status.key != "closed"


def is_issue_exist(issue_key):
    try:
        Client.issues[issue_key]
    except Exception:
        return False
    return True


@retrying.retry(stop_max_attempt_number=4, wait_fixed=4000)
def add_link(issue_key_parent, issue_key_child, relationship="relates"):
    issue = Client.issues[issue_key_parent]
    link = issue.links.create(issue=issue_key_child, relationship=relationship)


def find_components(queue, component_names):
    queue_components = Client.queues[queue].components
    result = []
    for component in queue_components:
        if component.name in component_names:
            result.append(component)
    return result
