package cachedresolver

import (
	"context"
	"encoding/json"
	"net"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/stretchr/testify/assert"

	pb "a.yandex-team.ru/infra/yp_service_discovery/api"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver/cacher"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver/httpresolver"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/httputil/headers"
)

func TestCachedResolver_Close(t *testing.T) {
	r, err := httpresolver.New()
	assert.NoError(t, err)

	c := new(cacher.Nop)

	cr, err := New(r, c)
	assert.NoError(t, err)
	assert.NoError(t, cr.Close())
}

func TestNew(t *testing.T) {
	r, err := httpresolver.New()
	assert.NoError(t, err)

	c := new(cacher.Nop)

	cr, err := New(r, c)
	assert.NoError(t, err)

	expected := &CachedResolver{
		resolver:    r,
		cache:       c,
		returnStale: false,
		logger:      new(nop.Logger),
	}

	opts := cmp.Options{
		cmp.AllowUnexported(CachedResolver{}),
		cmpopts.IgnoreUnexported(httpresolver.Resolver{}, cacher.InMemory{}),
	}

	assert.True(t, cmp.Equal(expected, cr, opts...), cmp.Diff(expected, cr, opts...))
}

func TestCachedResolver_ResolveEndpoints(t *testing.T) {
	now := time.Now()

	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		resp, err := json.Marshal(pb.TRspResolveEndpoints{
			Timestamp: uint64(now.Unix()),
			EndpointSet: &pb.TEndpointSet{
				EndpointSetId: "trololo",
				Endpoints: []*pb.TEndpoint{
					{
						Id:                   "shimba-boomba",
						Protocol:             "TCP",
						Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
						Ip6Address:           "2a02:6b8:c08:afa8:10d:bd77:7f89:0",
						Port:                 8080,
						LabelSelectorResults: []string{"looken-tooken"},
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			WatchToken:    "0",
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	c := new(cacher.Nop)

	cr, err := New(r, c)
	assert.NoError(t, err)

	defer cr.Close()

	resp, err := cr.ResolveEndpoints(context.Background(), "sas", "ololo")
	assert.NoError(t, err)

	expected := &resolver.ResolveEndpointsResponse{
		Timestamp: uint64(now.Unix()),
		EndpointSet: &resolver.EndpointSet{
			ID: "trololo",
			Endpoints: []*resolver.Endpoint{
				{
					ID:       "shimba-boomba",
					Protocol: "TCP",
					FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
					IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
					Port:     8080,
					Labels:   []string{"looken-tooken"},
				},
			},
		},
		ResolveStatus: resolver.StatusEndpointOK,
		WatchToken:    "0",
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
}

func TestCachedResolver_ResolveEndpoints_Cached(t *testing.T) {
	now := time.Now()

	var requestsServed int
	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		defer func() { requestsServed++ }()

		resp, err := json.Marshal(pb.TRspResolveEndpoints{
			Timestamp: uint64(now.Unix()),
			EndpointSet: &pb.TEndpointSet{
				EndpointSetId: "trololo",
				Endpoints: []*pb.TEndpoint{
					{
						Id:                   "shimba-boomba",
						Protocol:             "TCP",
						Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
						Ip6Address:           "2a02:6b8:c08:afa8:10d:bd77:7f89:0",
						Port:                 8080,
						LabelSelectorResults: []string{"looken-tooken"},
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			WatchToken:    "0",
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	c := cacher.NewInMemory()

	cr, err := New(r, c)
	assert.NoError(t, err)

	defer cr.Close()

	expected := &resolver.ResolveEndpointsResponse{
		Timestamp: uint64(now.Unix()),
		EndpointSet: &resolver.EndpointSet{
			ID: "trololo",
			Endpoints: []*resolver.Endpoint{
				{
					ID:       "shimba-boomba",
					Protocol: "TCP",
					FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
					IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
					Port:     8080,
					Labels:   []string{"looken-tooken"},
				},
			},
		},
		ResolveStatus: resolver.StatusEndpointOK,
		WatchToken:    "0",
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	for i := 0; i < 5; i++ {
		// actual response
		resp, err := cr.ResolveEndpoints(context.Background(), "sas", "ololo")
		assert.NoError(t, err)
		assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
	}

	srv.Close()
	assert.Equal(t, 1, requestsServed)
}

func TestCachedResolver_ResolveEndpoints_Stale(t *testing.T) {
	now := time.Now()

	var requestsServed int
	var srv *httptest.Server
	srv = httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		defer func() { requestsServed++ }()

		// response with error on any request but first
		if requestsServed > 0 {
			if requestsServed%2 == 0 {
				srv.CloseClientConnections()
			} else {
				w.WriteHeader(http.StatusInternalServerError)
			}
			return
		}

		resp, err := json.Marshal(pb.TRspResolveEndpoints{
			Timestamp: uint64(now.Unix()),
			EndpointSet: &pb.TEndpointSet{
				EndpointSetId: "trololo",
				Endpoints: []*pb.TEndpoint{
					{
						Id:                   "shimba-boomba",
						Protocol:             "TCP",
						Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
						Ip6Address:           "2a02:6b8:c08:afa8:10d:bd77:7f89:0",
						Port:                 8080,
						LabelSelectorResults: []string{"looken-tooken"},
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			WatchToken:    "0",
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	// deliberately set tiny fraction as TTL of time to be sure that
	// object expired at the time of any actual seek will retrieve it from cache
	c := cacher.NewInMemory(cacher.InMemoryObjectTTL(1 * time.Nanosecond))

	cr, err := New(r, c, ReturnStaleCache(true))
	assert.NoError(t, err)

	defer cr.Close()

	expected := &resolver.ResolveEndpointsResponse{
		Timestamp: uint64(now.Unix()),
		EndpointSet: &resolver.EndpointSet{
			ID: "trololo",
			Endpoints: []*resolver.Endpoint{
				{
					ID:       "shimba-boomba",
					Protocol: "TCP",
					FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
					IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
					Port:     8080,
					Labels:   []string{"looken-tooken"},
				},
			},
		},
		ResolveStatus: resolver.StatusEndpointOK,
		WatchToken:    "0",
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	for i := 0; i < 5; i++ {
		// actual response
		resp, err := cr.ResolveEndpoints(context.Background(), "sas", "ololo")
		assert.NoError(t, err)
		assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
	}

	srv.Close()
	assert.Equal(t, 5, requestsServed)
}

func TestCachedResolver_ResolvePods(t *testing.T) {
	now := time.Now()

	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		resp, err := json.Marshal(pb.TRspResolvePods{
			Timestamp: uint64(now.Unix()),
			PodSet: &pb.TPodSet{
				PodSetId: "sas-yt-seneca-sas-nodes-over-yp",
				Pods: []*pb.TPod{
					{
						Id:     "sas2-9558-node-seneca-sas",
						NodeId: "sas2-9558.search.yandex.net",
						Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
							{
								Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
								VlanId:         "fastbone",
								PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							},
						},
						Dns: &pb.TPod_TDns{
							PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	c := new(cacher.Nop)

	cr, err := New(r, c)
	assert.NoError(t, err)

	defer cr.Close()

	resp, err := cr.ResolvePods(context.Background(), "sas", "ololo")
	assert.NoError(t, err)

	expected := &resolver.ResolvePodsResponse{
		Timestamp: uint64(now.Unix()),
		PodSet: &resolver.PodSet{
			ID: "sas-yt-seneca-sas-nodes-over-yp",
			Pods: []*resolver.Pod{
				{
					ID:     "sas2-9558-node-seneca-sas",
					NodeID: "sas2-9558.search.yandex.net",
					IP6AddressAllocations: []resolver.IP6AddressAllocation{
						{
							Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
							VLANID:         "fastbone",
							PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
					DNS: &resolver.DNS{
						PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
			},
		},
		ResolveStatus: resolver.StatusPodOK,
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
}

func TestCachedResolver_ResolvePods_Cached(t *testing.T) {
	now := time.Now()

	var requestsServed int
	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		defer func() { requestsServed++ }()

		resp, err := json.Marshal(pb.TRspResolvePods{
			Timestamp: uint64(now.Unix()),
			PodSet: &pb.TPodSet{
				PodSetId: "sas-yt-seneca-sas-nodes-over-yp",
				Pods: []*pb.TPod{
					{
						Id:     "sas2-9558-node-seneca-sas",
						NodeId: "sas2-9558.search.yandex.net",
						Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
							{
								Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
								VlanId:         "fastbone",
								PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							},
						},
						Dns: &pb.TPod_TDns{
							PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	c := cacher.NewInMemory()

	cr, err := New(r, c)
	assert.NoError(t, err)

	defer cr.Close()

	expected := &resolver.ResolvePodsResponse{
		Timestamp: uint64(now.Unix()),
		PodSet: &resolver.PodSet{
			ID: "sas-yt-seneca-sas-nodes-over-yp",
			Pods: []*resolver.Pod{
				{
					ID:     "sas2-9558-node-seneca-sas",
					NodeID: "sas2-9558.search.yandex.net",
					IP6AddressAllocations: []resolver.IP6AddressAllocation{
						{
							Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
							VLANID:         "fastbone",
							PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
					DNS: &resolver.DNS{
						PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
			},
		},
		ResolveStatus: resolver.StatusPodOK,
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	for i := 0; i < 5; i++ {
		// actual response
		resp, err := cr.ResolvePods(context.Background(), "sas", "ololo")
		assert.NoError(t, err)
		assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
	}

	srv.Close()
	assert.Equal(t, 1, requestsServed)
}

func TestCachedResolver_ResolvePods_Stale(t *testing.T) {
	now := time.Now()

	var requestsServed int
	var srv *httptest.Server
	srv = httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		defer func() { requestsServed++ }()

		// response with error on any request but first
		if requestsServed > 0 {
			if requestsServed%2 == 0 {
				srv.CloseClientConnections()
			} else {
				w.WriteHeader(http.StatusInternalServerError)
			}
			return
		}

		resp, err := json.Marshal(pb.TRspResolvePods{
			Timestamp: uint64(now.Unix()),
			PodSet: &pb.TPodSet{
				PodSetId: "sas-yt-seneca-sas-nodes-over-yp",
				Pods: []*pb.TPod{
					{
						Id:     "sas2-9558-node-seneca-sas",
						NodeId: "sas2-9558.search.yandex.net",
						Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
							{
								Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
								VlanId:         "fastbone",
								PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							},
						},
						Dns: &pb.TPod_TDns{
							PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	// deliberately set tiny fraction as TTL of time to be sure that
	// object expired at the time of any actual seek will retrieve it from cache
	c := cacher.NewInMemory(cacher.InMemoryObjectTTL(1 * time.Nanosecond))

	cr, err := New(r, c, ReturnStaleCache(true))
	assert.NoError(t, err)

	defer cr.Close()

	expected := &resolver.ResolvePodsResponse{
		Timestamp: uint64(now.Unix()),
		PodSet: &resolver.PodSet{
			ID: "sas-yt-seneca-sas-nodes-over-yp",
			Pods: []*resolver.Pod{
				{
					ID:     "sas2-9558-node-seneca-sas",
					NodeID: "sas2-9558.search.yandex.net",
					IP6AddressAllocations: []resolver.IP6AddressAllocation{
						{
							Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
							VLANID:         "fastbone",
							PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
					DNS: &resolver.DNS{
						PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
			},
		},
		ResolveStatus: resolver.StatusPodOK,
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	for i := 0; i < 5; i++ {
		// actual response
		resp, err := cr.ResolvePods(context.Background(), "sas", "ololo")
		assert.NoError(t, err)
		assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
	}

	srv.Close()
	assert.Equal(t, 5, requestsServed)
}

func TestCachedResolver_ResolveNode(t *testing.T) {
	now := time.Now()

	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		resp, err := json.Marshal(pb.TRspResolveNode{
			Timestamp: uint64(now.Unix()),
			Pods: []*pb.TPod{
				{
					Id:     "sas2-9558-node-seneca-sas",
					NodeId: "sas2-9558.search.yandex.net",
					Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
						{
							Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
							VlanId:         "fastbone",
							PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
					Dns: &pb.TPod_TDns{
						PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	c := new(cacher.Nop)

	cr, err := New(r, c)
	assert.NoError(t, err)

	defer cr.Close()

	resp, err := cr.ResolveNode(context.Background(), "sas", "ololo")
	assert.NoError(t, err)

	expected := &resolver.ResolveNodeResponse{
		Timestamp: uint64(now.Unix()),
		Pods: []*resolver.Pod{
			{
				ID:     "sas2-9558-node-seneca-sas",
				NodeID: "sas2-9558.search.yandex.net",
				IP6AddressAllocations: []resolver.IP6AddressAllocation{
					{
						Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
						VLANID:         "fastbone",
						PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
				DNS: &resolver.DNS{
					PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
				},
			},
		},
		ResolveStatus: resolver.StatusPodOK,
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
}

func TestCachedResolver_Resolve_Cached(t *testing.T) {
	now := time.Now()

	var requestsServed int
	srv := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		defer func() { requestsServed++ }()

		resp, err := json.Marshal(pb.TRspResolveNode{
			Timestamp: uint64(now.Unix()),
			Pods: []*pb.TPod{
				{
					Id:     "sas2-9558-node-seneca-sas",
					NodeId: "sas2-9558.search.yandex.net",
					Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
						{
							Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
							VlanId:         "fastbone",
							PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
					Dns: &pb.TPod_TDns{
						PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	c := cacher.NewInMemory()

	cr, err := New(r, c)
	assert.NoError(t, err)

	defer cr.Close()

	expected := &resolver.ResolveNodeResponse{
		Timestamp: uint64(now.Unix()),
		Pods: []*resolver.Pod{
			{
				ID:     "sas2-9558-node-seneca-sas",
				NodeID: "sas2-9558.search.yandex.net",
				IP6AddressAllocations: []resolver.IP6AddressAllocation{
					{
						Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
						VLANID:         "fastbone",
						PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
				DNS: &resolver.DNS{
					PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
				},
			},
		},
		ResolveStatus: resolver.StatusPodOK,
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	for i := 0; i < 5; i++ {
		// actual response
		resp, err := cr.ResolveNode(context.Background(), "sas", "ololo")
		assert.NoError(t, err)
		assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
	}

	srv.Close()
	assert.Equal(t, 1, requestsServed)
}

func TestCachedResolver_ResolveNode_Stale(t *testing.T) {
	now := time.Now()

	var requestsServed int
	var srv *httptest.Server
	srv = httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		defer func() { requestsServed++ }()

		// response with error on any request but first
		if requestsServed > 0 {
			if requestsServed%2 == 0 {
				srv.CloseClientConnections()
			} else {
				w.WriteHeader(http.StatusInternalServerError)
			}
			return
		}

		resp, err := json.Marshal(pb.TRspResolveNode{
			Timestamp: uint64(now.Unix()),
			Pods: []*pb.TPod{
				{
					Id:     "sas2-9558-node-seneca-sas",
					NodeId: "sas2-9558.search.yandex.net",
					Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
						{
							Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
							VlanId:         "fastbone",
							PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
					Dns: &pb.TPod_TDns{
						PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
			},
			ResolveStatus: pb.EResolveStatus_OK,
			Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
			Ruid:          "ololo",
		})
		assert.NoError(t, err)

		w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
		_, err = w.Write(resp)
		assert.NoError(t, err)
	}))
	defer srv.Close()

	r, err := httpresolver.New(
		httpresolver.WithServiceURI(srv.URL),
	)
	assert.NoError(t, err)

	// deliberately set tiny fraction as TTL of time to be sure that
	// object expired at the time of any actual seek will retrieve it from cache
	c := cacher.NewInMemory(cacher.InMemoryObjectTTL(1 * time.Nanosecond))

	cr, err := New(r, c, ReturnStaleCache(true))
	assert.NoError(t, err)

	defer cr.Close()

	expected := &resolver.ResolveNodeResponse{
		Timestamp: uint64(now.Unix()),
		Pods: []*resolver.Pod{
			{
				ID:     "sas2-9558-node-seneca-sas",
				NodeID: "sas2-9558.search.yandex.net",
				IP6AddressAllocations: []resolver.IP6AddressAllocation{
					{
						Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
						VLANID:         "fastbone",
						PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					},
				},
				DNS: &resolver.DNS{
					PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
					TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
				},
			},
		},
		ResolveStatus: resolver.StatusPodOK,
		Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
		RUID:          "ololo",
	}

	for i := 0; i < 5; i++ {
		// actual response
		resp, err := cr.ResolveNode(context.Background(), "sas", "ololo")
		assert.NoError(t, err)
		assert.True(t, cmp.Equal(expected, resp), cmp.Diff(expected, resp))
	}

	srv.Close()
	assert.Equal(t, 5, requestsServed)
}
