package cacher

import (
	"errors"
	"time"

	"github.com/karlseguin/ccache/v2"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

var _ resolver.Cacher = new(InMemory)

// InMemory represents in-memory cache.
type InMemory struct {
	cache  *ccache.Cache
	logger log.Structured

	maxObjects int
	ttl        time.Duration
}

// NewInMemory creates instance of in-memory cacher.
// It is based on LRU cache with TTL.
// Example:
//    ttl := 1 * time.Minute
//    maxObjects := 2048
//    c := cacher.NewInMemory(
//        cacher.InMemoryObjectTTL(ttl),
//        cacher.MaxInMemoryObjects(maxObjects),
//    )
func NewInMemory(opts ...InMemoryOpt) *InMemory {
	cacher := &InMemory{
		maxObjects: 1024,
		ttl:        15 * time.Second,
		logger:     new(nop.Logger),
	}

	for _, opt := range opts {
		opt(cacher)
	}

	// create cache instance if none presents already
	if cacher.cache == nil {
		cacher.cache = ccache.New(ccache.Configure().
			MaxSize(int64(cacher.maxObjects)),
		)
	}

	return cacher
}

// Close stops underlying cache allowing GC to free memory.
func (c InMemory) Close() error {
	c.cache.Stop()
	return nil
}

// Get returns YP SD response object from cache.
func (c InMemory) Get(key string) (response interface{}, stale bool) {
	item := c.cache.Get(key)

	if item == nil {
		return nil, false
	}

	return item.Value(), item.Expired()
}

// Set writes YP SD response object to cache.
func (c InMemory) Set(key string, value interface{}) error {
	if key == "" {
		return errors.New("empty key given")
	}

	c.cache.Set(key, value, c.ttl)
	return nil
}

// InMemoryOpt allows to configure InMemory cacher.
type InMemoryOpt func(*InMemory)

// MaxInMemoryObjects set maximum number of objects in cache.
func MaxInMemoryObjects(max int) InMemoryOpt {
	return func(c *InMemory) {
		c.maxObjects = max
	}
}

// InMemoryObjectTTL sets time-to-live of object in cache.
func InMemoryObjectTTL(ttl time.Duration) InMemoryOpt {
	return func(c *InMemory) {
		c.ttl = ttl
	}
}

// InMemoryLogger sets logger for in-memory cacher.
func InMemoryLogger(l log.Structured) InMemoryOpt {
	return func(c *InMemory) {
		c.logger = l
	}
}
