package cacher

import (
	"net"
	"testing"
	"time"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/karlseguin/ccache/v2"
	"github.com/stretchr/testify/assert"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/core/log/zap"
)

func TestNewInMemory(t *testing.T) {
	c := NewInMemory()

	expected := &InMemory{
		maxObjects: 1024,
		ttl:        15 * time.Second,
		logger:     new(nop.Logger),
		cache: ccache.New(ccache.Configure().
			MaxSize(int64(1024)),
		),
	}

	opts := cmp.Options{
		cmp.AllowUnexported(InMemory{}, ccache.Configuration{}),
		cmpopts.IgnoreUnexported(ccache.Cache{}),
	}

	assert.True(t, cmp.Equal(expected, c, opts...), cmp.Diff(expected, c, opts...))
}

func TestMaxInMemoryObjects(t *testing.T) {
	maxObjects := 42
	c := NewInMemory(MaxInMemoryObjects(maxObjects))
	assert.Equal(t, maxObjects, c.maxObjects)
}

func TestInMemoryObjectTTL(t *testing.T) {
	ttl := 1 * time.Minute
	c := NewInMemory(InMemoryObjectTTL(ttl))
	assert.Equal(t, ttl, c.ttl)
}

func TestInMemoryLogger(t *testing.T) {
	logger := zap.Must(zap.CLIConfig(log.DebugLevel))
	c := NewInMemory(InMemoryLogger(logger))
	assert.Equal(t, logger, c.logger)
}

func TestInMemory_Close(t *testing.T) {
	c := NewInMemory()
	err := c.Close()
	assert.NoError(t, err)
	assert.Equal(t, 0, c.cache.ItemCount())
}

func TestInMemory_Set(t *testing.T) {
	c := NewInMemory()

	value := &resolver.ResolveEndpointsResponse{
		ResolveStatus: resolver.StatusEndpointOK,
		Host:          "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
		RUID:          "trololo",
		WatchToken:    "0",
		EndpointSet: &resolver.EndpointSet{
			ID: "ololo",
			Endpoints: []*resolver.Endpoint{
				{
					ID:       "0k9aa7s7ucpa2ez3",
					Protocol: "TCP",
					FQDN:     "nbiddl742lszft2y.sas.yp-c.yandex.net",
					IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
					Port:     3388,
				},
			},
		},
	}

	err := c.Set("ololo", value)
	assert.NoError(t, err)

	item := c.cache.Get("ololo")
	assert.NotNil(t, item)
	assert.Same(t, value, item.Value())
}

func TestInMemory_Get(t *testing.T) {
	c := NewInMemory()

	value := &resolver.ResolveEndpointsResponse{
		ResolveStatus: resolver.StatusEndpointOK,
		Host:          "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
		RUID:          "trololo",
		WatchToken:    "0",
		EndpointSet: &resolver.EndpointSet{
			ID: "ololo",
			Endpoints: []*resolver.Endpoint{
				{
					ID:       "0k9aa7s7ucpa2ez3",
					Protocol: "TCP",
					FQDN:     "nbiddl742lszft2y.sas.yp-c.yandex.net",
					IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
					Port:     3388,
				},
			},
		},
	}

	c.cache.Set("ololo", value, c.ttl)

	cached, stale := c.Get("ololo")
	assert.False(t, stale)
	assert.Same(t, value, cached)
}

func TestInMemory_Expiration(t *testing.T) {
	c := NewInMemory(InMemoryObjectTTL(1 * time.Second))

	value := &resolver.ResolveEndpointsResponse{
		ResolveStatus: resolver.StatusEndpointOK,
		Host:          "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
		RUID:          "trololo",
		WatchToken:    "0",
		EndpointSet: &resolver.EndpointSet{
			ID: "ololo",
			Endpoints: []*resolver.Endpoint{
				{
					ID:       "0k9aa7s7ucpa2ez3",
					Protocol: "TCP",
					FQDN:     "nbiddl742lszft2y.sas.yp-c.yandex.net",
					IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
					Port:     3388,
				},
			},
		},
	}

	err := c.Set("ololo", value)
	assert.NoError(t, err)

	// check cached
	cached, stale := c.Get("ololo")
	assert.False(t, stale)
	assert.Same(t, value, cached)

	// check expired
	time.Sleep(2 * time.Second)
	expired, stale := c.Get("ololo")
	assert.True(t, stale)
	assert.Same(t, value, expired)
}
