package cacher

import (
	"bytes"
	"encoding/gob"
	"fmt"
	"sync"
	"time"

	"go.etcd.io/bbolt"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/core/xerrors"
)

var (
	onDiskDefaultBucket = []byte("cache")
)

var onDiskOnce sync.Once

var _ resolver.Cacher = new(OnDisk)

// OnDisk represents persistent filesystem cache.
type OnDisk struct {
	logger log.Structured
	cache  *bbolt.DB

	path string

	ttl time.Duration
}

// NewOnDisk creates instance of on-disk cacher.
// Example:
//    ttl := 1 * time.Minute
//    c := cacher.NewOnDisk(
//        cacher.OnDiskBaseDir("/var/cache/ypsd"),
//        cacher.OnDiskObjectTTL(ttl),
//    )
func NewOnDisk(opts ...OnDiskOpt) (*OnDisk, error) {
	// register resolver response as gob type
	onDiskOnce.Do(func() {
		gob.Register(onDiskCacheRecord{})
		gob.Register(&resolver.ResolveEndpointsResponse{})
		gob.Register(&resolver.ResolvePodsResponse{})
	})

	c := &OnDisk{
		logger: new(nop.Logger),
		path:   "cache.bin",
		ttl:    15 * time.Second,
	}

	for _, opt := range opts {
		opt(c)
	}

	if c.cache == nil {
		db, err := bbolt.Open(c.path, 0600, &bbolt.Options{Timeout: 1 * time.Second})
		if err != nil {
			return nil, err
		}
		c.cache = db
	}

	return c, nil
}

// Close removes all cache.
func (c *OnDisk) Close() error {
	return c.cache.Close()
}

// Get returns YP SD response object from cache.
// Expired files will be deleted upon request in any way,
// but response would be returned once before deletion if
// ReturnStaleFromDisk option enabled.
func (c OnDisk) Get(key string) (response interface{}, stale bool) {
	err := c.cache.View(func(tx *bbolt.Tx) error {
		bucket := tx.Bucket(onDiskDefaultBucket)
		if bucket == nil {
			return bbolt.ErrBucketNotFound
		}

		b := bucket.Get([]byte(key))
		if b == nil {
			return fmt.Errorf("key not found: %s", key)
		}

		var rec onDiskCacheRecord
		err := gob.NewDecoder(bytes.NewBuffer(b)).Decode(&rec)
		if err != nil {
			return xerrors.Errorf("cannot decode cached object from gob: %w", err)
		}

		response = rec.Value
		stale = rec.Expired(c.ttl)
		return nil
	})

	if err != nil {
		return nil, false
	}

	return
}

// Set writes YP SD response object to cache.
// The data in underlying file will be encoded in GOB binary format.
func (c OnDisk) Set(key string, value interface{}) error {
	return c.cache.Update(func(tx *bbolt.Tx) error {
		bucket, err := tx.CreateBucketIfNotExists(onDiskDefaultBucket)
		if err != nil {
			return err
		}

		rec := onDiskCacheRecord{
			Mtime: time.Now(),
			Value: value,
		}

		var buf bytes.Buffer
		err = gob.NewEncoder(&buf).Encode(rec)
		if err != nil {
			return err
		}

		return bucket.Put([]byte(key), buf.Bytes())
	})
}

// This struct will be encoded in GOB format prior to storing into cache.
// Be careful about backward compatibility with existing cached objects.
type onDiskCacheRecord struct {
	Mtime time.Time
	Value interface{}
}

func (r onDiskCacheRecord) Expired(ttl time.Duration) bool {
	if ttl <= 0 {
		return false
	}
	return r.Mtime.Before(time.Now().Add(ttl * -1))
}

// OnDiskOpt allows to configure OnDisk cacher.
type OnDiskOpt func(*OnDisk)

// OnDiskPath sets cache file path.
func OnDiskPath(path string) OnDiskOpt {
	return func(c *OnDisk) {
		c.path = path
	}
}

// OnDiskObjectTTL sets time-to-live of object in cache.
func OnDiskObjectTTL(ttl time.Duration) OnDiskOpt {
	return func(c *OnDisk) {
		c.ttl = ttl
	}
}

// OnDiskLogger sets logger for on-disk cacher.
func OnDiskLogger(l log.Structured) OnDiskOpt {
	return func(c *OnDisk) {
		c.logger = l
	}
}
