package cacher

import (
	"bytes"
	"encoding/gob"
	"net"
	"os"
	"testing"
	"time"

	"github.com/gofrs/uuid"
	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"go.etcd.io/bbolt"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/core/log/zap"
)

func cleanupTestcache(path string) {
	_ = os.RemoveAll(path)
}

func TestNewOnDisk(t *testing.T) {
	defer cleanupTestcache("cache.bin")

	c, err := NewOnDisk()
	assert.NoError(t, err)

	defer c.cache.Close()

	expected := &OnDisk{
		logger: new(nop.Logger),
		path:   "cache.bin",
		ttl:    15 * time.Second,
	}

	opts := cmp.Options{
		cmp.AllowUnexported(OnDisk{}),
		cmpopts.IgnoreTypes(&bbolt.DB{}),
	}

	assert.True(t, cmp.Equal(expected, c, opts...), cmp.Diff(expected, c, opts...))
}

func TestOnDisk_Close(t *testing.T) {
	defer cleanupTestcache("cache.bin")

	c, err := NewOnDisk()
	assert.NoError(t, err)

	err = c.Close()
	assert.NoError(t, err)

	assert.Panics(t, func() { c.cache.Info() })
}

func TestOnDiskPath(t *testing.T) {
	defer cleanupTestcache("testcache.bin")

	path := "testcache.bin"
	c, err := NewOnDisk(OnDiskPath(path))
	assert.NoError(t, err)
	assert.Equal(t, path, c.path)
}

func TestOnDiskObjectTTL(t *testing.T) {
	defer cleanupTestcache("cache.bin")

	logger := zap.Must(zap.CLIConfig(log.DebugLevel))
	c, err := NewOnDisk(OnDiskLogger(logger))
	assert.NoError(t, err)
	assert.Equal(t, c.logger, logger)
}

func TestOnDiskObjectTTL1(t *testing.T) {
	defer cleanupTestcache("cache.bin")

	ttl := 1 * time.Minute
	c, err := NewOnDisk(OnDiskObjectTTL(ttl))
	assert.NoError(t, err)
	assert.Equal(t, c.ttl, ttl)
}

func TestOnDisk_Set(t *testing.T) {
	defer cleanupTestcache("cache.bin")

	c, err := NewOnDisk()
	assert.NoError(t, err)

	value := &resolver.ResolveEndpointsResponse{
		ResolveStatus: resolver.StatusEndpointOK,
		Host:          "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
		RUID:          "trololo",
		WatchToken:    "0",
		EndpointSet: &resolver.EndpointSet{
			ID: "ololo",
			Endpoints: []*resolver.Endpoint{
				{
					ID:       "0k9aa7s7ucpa2ez3",
					Protocol: "TCP",
					FQDN:     "nbiddl742lszft2y.sas.yp-c.yandex.net",
					IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
					Port:     3388,
				},
			},
		},
	}

	err = c.Set("ololo", value)
	assert.NoError(t, err)

	var rec onDiskCacheRecord
	err = c.cache.View(func(tx *bbolt.Tx) error {
		bucket := tx.Bucket(onDiskDefaultBucket)
		require.NotNil(t, bucket)

		b := bucket.Get([]byte("ololo"))
		assert.NotEmpty(t, b)

		return gob.NewDecoder(bytes.NewBuffer(b)).Decode(&rec)
	})
	assert.NoError(t, err)

	assert.True(t, cmp.Equal(value, rec.Value), cmp.Diff(value, rec.Value))
}

func TestOnDisk_Get(t *testing.T) {
	defer cleanupTestcache("cache.bin")

	c, err := NewOnDisk()
	assert.NoError(t, err)

	// try uncached
	found, stale := c.Get(uuid.Must(uuid.NewV4()).String())
	assert.Nil(t, found)
	assert.False(t, stale)

	for _, tc := range []struct {
		name  string
		key   string
		value interface{}
	}{
		{
			name: "endpoints",
			key:  "ololo",
			value: &resolver.ResolveEndpointsResponse{
				ResolveStatus: resolver.StatusEndpointOK,
				Host:          "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
				RUID:          "trololo",
				WatchToken:    "0",
				EndpointSet: &resolver.EndpointSet{
					ID: "ololo",
					Endpoints: []*resolver.Endpoint{
						{
							ID:       "0k9aa7s7ucpa2ez3",
							Protocol: "TCP",
							FQDN:     "nbiddl742lszft2y.sas.yp-c.yandex.net",
							IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
							Port:     3388,
						},
					},
				},
			},
		},
		{
			name: "pods",
			key:  "sas-yt-seneca-sas-nodes-over-yp",
			value: &resolver.ResolvePodsResponse{
				ResolveStatus: resolver.StatusPodOK,
				Host:          "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
				RUID:          "trololo",
				PodSet: &resolver.PodSet{
					ID: "sas-yt-seneca-sas-nodes-over-yp",
					Pods: []*resolver.Pod{
						{
							ID:     "sas2-9558-node-seneca-sas",
							NodeID: "sas2-9558.search.yandex.net",
							IP6AddressAllocations: []resolver.IP6AddressAllocation{
								{
									Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
									VLANID:         "fastbone",
									PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
									TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								},
							},
							DNS: &resolver.DNS{
								PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							},
						},
					},
				},
			},
		},
	} {
		t.Run(tc.name, func(t *testing.T) {
			err = c.Set(tc.key, tc.value)
			assert.NoError(t, err)

			found, stale = c.Get(tc.key)
			assert.False(t, stale)
			assert.True(t, cmp.Equal(tc.value, found), cmp.Diff(tc.value, found))
		})
	}
}

func TestOnDisk_Expiration(t *testing.T) {
	defer cleanupTestcache("cache.bin")

	c, err := NewOnDisk(OnDiskObjectTTL(1 * time.Second))
	assert.NoError(t, err)

	key := "ololo"
	response := &resolver.ResolveEndpointsResponse{
		ResolveStatus: resolver.StatusEndpointNotExists,
		Host:          "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
		RUID:          "trololo",
	}

	err = c.Set(key, response)
	assert.NoError(t, err)

	cachedResponse, stale := c.Get(key)
	assert.False(t, stale)
	assert.True(t, cmp.Equal(response, cachedResponse), cmp.Diff(response, cachedResponse))

	// wait till object expires
	time.Sleep(2 * time.Second)

	expiredResponse, stale := c.Get(key)
	assert.True(t, stale)
	assert.True(t, cmp.Equal(response, expiredResponse), cmp.Diff(response, expiredResponse))
}
