package grpcresolver

import (
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"

	pb "a.yandex-team.ru/infra/yp_service_discovery/api"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

func TestWithClientName(t *testing.T) {
	clientName := "volozh"
	r := new(Resolver)
	WithClientName(clientName)(r)
	assert.Equal(t, clientName, r.clientName)
}

func TestWithLogger(t *testing.T) {
	logger := zap.Must(zap.CLIConfig(log.FatalLevel))
	r := new(Resolver)
	WithLogger(logger)(r)
	assert.Same(t, logger, r.logger)
}

func TestWithServiceURI(t *testing.T) {
	serviceURI := "sd-test.yandex.net"
	r := new(Resolver)
	WithServiceURI(serviceURI)(r)
	assert.Equal(t, serviceURI, r.serviceURI)
}

func TestWithResolveEndpointsRUIDFunc(t *testing.T) {
	fn := func(_ *pb.TReqResolveEndpoints) string {
		return "ololo"
	}
	r := new(Resolver)
	WithResolveEndpointsRUIDFunc(fn)(r)
	assert.Equal(t, fmt.Sprintf("%p", fn), fmt.Sprintf("%p", r.resolveEndpointsRUIDFunc))
}

func TestWithResolvePodsRUIDFunc(t *testing.T) {
	fn := func(_ *pb.TReqResolvePods) string {
		return "ololo"
	}
	r := new(Resolver)
	WithResolvePodsRUIDFunc(fn)(r)
	assert.Equal(t, fmt.Sprintf("%p", fn), fmt.Sprintf("%p", r.resolvePodsRUIDFunc))
}

func TestWithResolveNodeRUIDFunc(t *testing.T) {
	fn := func(_ *pb.TReqResolveNode) string {
		return "ololo"
	}
	r := new(Resolver)
	WithResolveNodeRUIDFunc(fn)(r)
	assert.Equal(t, fmt.Sprintf("%p", fn), fmt.Sprintf("%p", r.resolveNodeRUIDFunc))
}

func TestWithGRPCClient(t *testing.T) {
	srv, client, conn, lis := newTestYPSDService(t, nil)
	defer lis.Close()
	defer srv.Stop()
	defer conn.Close()

	r := new(Resolver)
	WithGRPCClient(client, conn)(r)
	assert.Same(t, client, r.grpcClient)
	assert.Same(t, conn, r.grpcConn)
}

func TestWithGRPCConn(t *testing.T) {
	srv, _, conn, lis := newTestYPSDService(t, nil)
	defer func() { _ = lis.Close() }()
	defer srv.Stop()
	defer func() { _ = conn.Close() }()

	r := new(Resolver)
	WithGRPCConn(conn)(r)
	assert.Same(t, conn, r.grpcConn)
}
