package httpresolver

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"os"

	"github.com/go-resty/resty/v2"
	"github.com/gofrs/uuid"

	pb "a.yandex-team.ru/infra/yp_service_discovery/api"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/httputil/headers"
)

var _ io.Closer = new(Resolver)
var _ resolver.Resolver = new(Resolver)

type ResolveEndpointsRUIDFunc func(*pb.TReqResolveEndpoints) string
type ResolvePodsRUIDFunc func(pods *pb.TReqResolvePods) string
type ResolveNodeRUIDFunc func(*pb.TReqResolveNode) string

type Resolver struct {
	serviceURI string
	clientName string

	httpc                    *resty.Client
	resolveEndpointsRUIDFunc ResolveEndpointsRUIDFunc
	resolvePodsRUIDFunc      ResolvePodsRUIDFunc
	resolveNodeRUIDFunc      ResolveNodeRUIDFunc

	logger log.Structured
}

// New returns HTTP resolver instance.
func New(opts ...ResolverOpt) (*Resolver, error) {
	r := &Resolver{
		serviceURI: "http://" + resolver.ServiceDiscoveryHostProd + ":" + resolver.ServiceDiscoveryHTTPPort,
		clientName: getClientName(),
		logger:     new(nop.Logger),
		resolveEndpointsRUIDFunc: func(_ *pb.TReqResolveEndpoints) string {
			return uuid.Must(uuid.NewV4()).String()
		},
		resolvePodsRUIDFunc: func(_ *pb.TReqResolvePods) string {
			return uuid.Must(uuid.NewV4()).String()
		},
		resolveNodeRUIDFunc: func(_ *pb.TReqResolveNode) string {
			return uuid.Must(uuid.NewV4()).String()
		},
	}

	for _, opt := range opts {
		opt(r)
	}

	// create HTTP client if none presents already
	if r.httpc == nil {
		r.httpc = resty.New().
			SetHostURL(r.serviceURI).
			SetLogger(r.logger.Fmt())
	}

	r.logger.Debug("new resolver instance created", log.Any("resolver", r))
	return r, nil
}

func (r *Resolver) Close() error {
	r.logger.Debug("closing YP SD HTTP resolver")
	r.httpc = nil
	return nil
}

func (r Resolver) ResolveEndpoints(ctx context.Context, cluster, endpointSet string) (*resolver.ResolveEndpointsResponse, error) {
	request := &pb.TReqResolveEndpoints{
		ClusterName:   cluster,
		EndpointSetId: endpointSet,
		ClientName:    r.clientName,
	}
	request.Ruid = r.resolveEndpointsRUIDFunc(request)
	r.logger.Debug("sending resolve request", log.Any("request", request))

	res, err := r.httpc.R().
		SetContext(ctx).
		SetHeader(headers.ContentTypeKey, headers.TypeApplicationJSON.String()).
		SetBody(request).
		Post("/resolve_endpoints/json")

	r.logger.Debug("got resolving response",
		log.Any("response", res),
		log.Error(err),
	)

	if err != nil {
		return nil, xerrors.Errorf("failed to resolve endpoint-set %s@%s: %w", endpointSet, cluster, err)
	}

	if res.IsError() {
		return nil, fmt.Errorf("unsupported status code: %d", res.StatusCode())
	}

	var response resolver.ResolveEndpointsResponse
	err = json.Unmarshal(res.Body(), &response)
	if err != nil {
		return nil, fmt.Errorf("cannot unmarshal body: %w", err)
	}

	return &response, nil
}

func (r *Resolver) ResolvePods(ctx context.Context, cluster, podSet string) (*resolver.ResolvePodsResponse, error) {
	request := &pb.TReqResolvePods{
		ClusterName: cluster,
		PodSetId:    podSet,
		ClientName:  r.clientName,
	}
	request.Ruid = r.resolvePodsRUIDFunc(request)
	r.logger.Debug("sending resolve request", log.Any("request", request))

	res, err := r.httpc.R().
		SetContext(ctx).
		SetHeader(headers.ContentTypeKey, headers.TypeApplicationJSON.String()).
		SetBody(request).
		Post("/resolve_pods/json")

	r.logger.Debug("got resolving response",
		log.Any("response", res),
		log.Error(err),
	)

	if err != nil {
		return nil, xerrors.Errorf("failed to resolve pod-set %s@%s: %w", podSet, cluster, err)
	}

	if res.IsError() {
		return nil, fmt.Errorf("unsupported status code: %d", res.StatusCode())
	}

	var response resolver.ResolvePodsResponse
	err = json.Unmarshal(res.Body(), &response)
	if err != nil {
		return nil, fmt.Errorf("cannot unmarshal body: %w", err)
	}

	return &response, nil
}

func (r *Resolver) ResolveNode(ctx context.Context, cluster, node string) (*resolver.ResolveNodeResponse, error) {
	request := &pb.TReqResolveNode{
		ClusterName: cluster,
		NodeId:      node,
		ClientName:  r.clientName,
	}
	request.Ruid = r.resolveNodeRUIDFunc(request)
	r.logger.Debug("sending resolve request", log.Any("request", request))

	res, err := r.httpc.R().
		SetContext(ctx).
		SetHeader(headers.ContentTypeKey, headers.TypeApplicationJSON.String()).
		SetBody(request).
		Post("/resolve_node/json")

	r.logger.Debug("got resolving response",
		log.Any("response", res),
		log.Error(err),
	)

	if err != nil {
		return nil, xerrors.Errorf("failed to resolve node %s@%s: %w", node, cluster, err)
	}

	if res.IsError() {
		return nil, fmt.Errorf("unsupported status code: %d", res.StatusCode())
	}

	var response resolver.ResolveNodeResponse
	err = json.Unmarshal(res.Body(), &response)
	if err != nil {
		return nil, fmt.Errorf("cannot unmarshal body: %w", err)
	}

	return &response, nil
}

func getClientName() string {
	hostname := "unknown"
	if osHostname, err := os.Hostname(); err == nil {
		hostname = osHostname
	}

	username := "go_resolver"
	for _, key := range []string{"SUDO_USER", "USER"} {
		if user := os.Getenv(key); user != "" {
			username = user
			break
		}
	}

	return username + "@" + hostname
}
