package resolver

import (
	"fmt"
	"net"

	pb "a.yandex-team.ru/infra/yp_service_discovery/api"
)

const (
	StatusEndpointNotExists = int(pb.EResolveStatus_NOT_EXISTS)
	StatusEndpointOK        = int(pb.EResolveStatus_OK)
	StatusEndpointEmpty     = int(pb.EResolveStatus_EMPTY)
	// Deprecated: this status is deprecated as per https://nda.ya.ru/t/HpRd8Ys83dDhgx
	StatusEndpointNotChanged = int(pb.EResolveStatus_NOT_CHANGED)

	StatusPodNotExists = int(pb.EResolveStatus_NOT_EXISTS)
	StatusPodOK        = int(pb.EResolveStatus_OK)
	StatusPodEmpty     = int(pb.EResolveStatus_EMPTY)
)

type ResolveEndpointsResponse struct {
	Timestamp     uint64       `json:"timestamp"`
	EndpointSet   *EndpointSet `json:"endpoint_set"`
	ResolveStatus int          `json:"resolve_status"`
	WatchToken    string       `json:"watch_token"`
	Host          string       `json:"host"`
	RUID          string       `json:"ruid"`
}

func (r *ResolveEndpointsResponse) UnmarshalProto(a *pb.TRspResolveEndpoints) error {
	if a.EndpointSet != nil {
		var es EndpointSet
		if err := es.UnmarshalProto(a.EndpointSet); err != nil {
			return err
		}
		r.EndpointSet = &es
	}

	r.Timestamp = a.Timestamp
	r.WatchToken = a.WatchToken
	r.Host = a.Host
	r.RUID = a.Ruid
	r.ResolveStatus = int(a.ResolveStatus)

	return nil
}

type EndpointSet struct {
	ID        string      `json:"endpoint_set_id"`
	Endpoints []*Endpoint `json:"endpoints"`
}

func (e *EndpointSet) UnmarshalProto(a *pb.TEndpointSet) error {
	endpoints := make([]*Endpoint, 0, len(a.Endpoints))
	for _, pbEndpoint := range a.Endpoints {
		var endpoint Endpoint
		if err := endpoint.UnmarshalProto(pbEndpoint); err != nil {
			return err
		}
		endpoints = append(endpoints, &endpoint)
	}

	e.ID = a.EndpointSetId
	e.Endpoints = endpoints

	return nil
}

type Endpoint struct {
	ID       string   `json:"id"`
	Protocol string   `json:"protocol"`
	FQDN     string   `json:"fqdn"`
	IPv4     net.IP   `json:"ip4_address"`
	IPv6     net.IP   `json:"ip6_address"`
	Port     int32    `json:"port"`
	Labels   []string `json:"label_selector_results"`
	Ready    bool     `json:"ready"`
}

func (e *Endpoint) UnmarshalProto(a *pb.TEndpoint) error {
	e.IPv4 = net.ParseIP(a.Ip4Address)
	e.IPv6 = net.ParseIP(a.Ip6Address)

	if e.IPv4 == nil && e.IPv6 == nil {
		return fmt.Errorf("endpoint '%s' has no valid IP addresses", a.Fqdn)
	}

	e.ID = a.Id
	e.Protocol = a.Protocol
	e.FQDN = a.Fqdn
	e.Port = a.Port
	e.Labels = a.LabelSelectorResults
	e.Ready = a.Ready

	return nil
}

type ResolveNodeResponse struct {
	Timestamp     uint64 `json:"timestamp"`
	Pods          []*Pod `json:"pods"`
	ResolveStatus int    `json:"resolve_status"`
	Host          string `json:"host"`
	RUID          string `json:"ruid"`
}

func (r *ResolveNodeResponse) UnmarshalProto(a *pb.TRspResolveNode) error {
	pods := make([]*Pod, 0, len(a.Pods))
	for _, pbPod := range a.Pods {
		var pod Pod
		if err := pod.UnmarshalProto(pbPod); err != nil {
			return err
		}
		pods = append(pods, &pod)
	}

	r.Pods = pods
	r.Timestamp = a.Timestamp
	r.Host = a.Host
	r.RUID = a.Ruid
	r.ResolveStatus = int(a.ResolveStatus)

	return nil
}

type ResolvePodsResponse struct {
	Timestamp     uint64  `json:"timestamp"`
	PodSet        *PodSet `json:"pod_set"`
	ResolveStatus int     `json:"resolve_status"`
	Host          string  `json:"host"`
	RUID          string  `json:"ruid"`
}

func (r *ResolvePodsResponse) UnmarshalProto(a *pb.TRspResolvePods) error {
	if a.PodSet != nil {
		var ps PodSet
		if err := ps.UnmarshalProto(a.PodSet); err != nil {
			return err
		}
		r.PodSet = &ps
	}

	r.Timestamp = a.Timestamp
	r.Host = a.Host
	r.RUID = a.Ruid
	r.ResolveStatus = int(a.ResolveStatus)

	return nil
}

type PodSet struct {
	ID   string `json:"pod_set_id"`
	Pods []*Pod `json:"pods"`
}

func (e *PodSet) UnmarshalProto(a *pb.TPodSet) error {
	pods := make([]*Pod, 0, len(a.Pods))
	for _, pbPod := range a.Pods {
		var pod Pod
		if err := pod.UnmarshalProto(pbPod); err != nil {
			return err
		}
		pods = append(pods, &pod)
	}

	e.ID = a.PodSetId
	e.Pods = pods

	return nil
}

type IssConfSummary struct {
	TargetState string `json:"target_state"`
}

func (s *IssConfSummary) UnmarshalProto(a *pb.TPod_TIssConfSummary) error {
	s.TargetState = a.TargetState

	return nil
}

type Pod struct {
	ID     string `json:"id"`
	NodeID string `json:"node_id"`

	IP6AddressAllocations []IP6AddressAllocation `json:"ip6_address_allocations"`
	IP6SubnetAllocations  []IP6SubnetAllocation  `json:"ip6_subnet_allocations"`
	DNS                   *DNS                   `json:"dns"`

	IssConfSummaries map[string]IssConfSummary `json:"iss_conf_summaries"`
}

func (p *Pod) UnmarshalProto(a *pb.TPod) error {
	p.ID = a.Id
	p.NodeID = a.NodeId

	addressAllocations := make([]IP6AddressAllocation, 0, len(a.Ip6AddressAllocations))
	for _, pbAlloc := range a.Ip6AddressAllocations {
		var alloc IP6AddressAllocation
		if err := alloc.UnmarshalProto(pbAlloc); err != nil {
			return err
		}
		addressAllocations = append(addressAllocations, alloc)
	}
	p.IP6AddressAllocations = addressAllocations

	subnetAllocations := make([]IP6SubnetAllocation, 0, len(a.Ip6SubnetAllocations))
	for _, pbAlloc := range a.Ip6SubnetAllocations {
		var alloc IP6SubnetAllocation
		if err := alloc.UnmarshalProto(pbAlloc); err != nil {
			return err
		}
		subnetAllocations = append(subnetAllocations, alloc)
	}
	p.IP6SubnetAllocations = subnetAllocations

	if a.Dns != nil {
		var dns DNS
		if err := dns.UnmarshalProto(a.Dns); err != nil {
			return err
		}
		p.DNS = &dns
	}

	if len(a.IssConfSummaries) > 0 {
		p.IssConfSummaries = make(map[string]IssConfSummary)
		for id, pbSummary := range a.IssConfSummaries {
			var summary IssConfSummary
			if err := summary.UnmarshalProto(pbSummary); err != nil {
				return err
			}
			p.IssConfSummaries[id] = summary
		}
	}

	return nil
}

type DNS struct {
	PersistentFQDN string `json:"persistent_fqdn"`
	TransientFQDN  string `json:"transient_fqdn"`
}

func (dns *DNS) UnmarshalProto(a *pb.TPod_TDns) error {
	dns.PersistentFQDN = a.PersistentFqdn
	dns.TransientFQDN = a.TransientFqdn

	return nil
}

type IP6AddressAllocation struct {
	Address         string           `json:"address"`
	VLANID          string           `json:"vlan_id"`
	PersistentFQDN  string           `json:"persistent_fqdn"`
	TransientFQDN   string           `json:"transient_fqdn"`
	InternetAddress *InternetAddress `json:"internet_address"`
	VirtualServices []VirtualService `json:"virtual_services"`
}

func (a *IP6AddressAllocation) UnmarshalProto(o *pb.TPod_TIP6AddressAllocation) error {
	a.Address = o.Address
	a.VLANID = o.VlanId
	a.PersistentFQDN = o.PersistentFqdn
	a.TransientFQDN = o.TransientFqdn

	if o.InternetAddress != nil {
		var inetAddr InternetAddress
		if err := inetAddr.UnmarshalProto(o.InternetAddress); err != nil {
			return err
		}
		a.InternetAddress = &inetAddr
	}

	virtualServices := make([]VirtualService, 0, len(o.VirtualServices))
	for _, pbService := range o.VirtualServices {
		var service VirtualService
		if err := service.UnmarshalProto(pbService); err != nil {
			return err
		}
		virtualServices = append(virtualServices, service)
	}
	a.VirtualServices = virtualServices

	return nil
}

type InternetAddress struct {
	ID         string `json:"id"`
	IP4Address string `json:"ip4_address"`
}

func (a *InternetAddress) UnmarshalProto(o *pb.TPod_TIP6AddressAllocation_TInternetAddress) error {
	a.ID = o.Id
	a.IP4Address = o.Ip4Address

	return nil
}

type VirtualService struct {
	IP6Addresses []string `json:"ip6_addresses"`
	IP4Addresses []string `json:"ip4_addresses"`
}

func (s *VirtualService) UnmarshalProto(o *pb.TPod_TIP6AddressAllocation_TVirtualService) error {
	s.IP6Addresses = o.Ip6Addresses
	s.IP4Addresses = o.Ip4Addresses

	return nil
}

type IP6SubnetAllocation struct {
	Subnet string `json:"subnet"`
	VLANID string `json:"vlan_id"`
}

func (a *IP6SubnetAllocation) UnmarshalProto(o *pb.TPod_TIP6SubnetAllocation) error {
	a.Subnet = o.Subnet
	a.VLANID = o.VlanId

	return nil
}
