package resolver

import (
	"errors"
	"net"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"

	pb "a.yandex-team.ru/infra/yp_service_discovery/api"
)

func TestResolveEndpointsResponse_UnmarshalProto(t *testing.T) {
	now := time.Now()

	testCases := []struct {
		name        string
		protoMsg    *pb.TRspResolveEndpoints
		expected    ResolveEndpointsResponse
		expectedErr error
	}{
		{
			name: "error_no_ips",
			protoMsg: &pb.TRspResolveEndpoints{
				Timestamp: uint64(now.Unix()),
				EndpointSet: &pb.TEndpointSet{
					EndpointSetId: "trololo",
					Endpoints: []*pb.TEndpoint{
						{
							Id:                   "shimba-boomba",
							Protocol:             "TCP",
							Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
							Port:                 8080,
							LabelSelectorResults: []string{"looken-tooken"},
							Ready:                true,
						},
					},
				},
				ResolveStatus: pb.EResolveStatus_OK,
				WatchToken:    "0",
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				Ruid:          "ololo",
			},
			expected:    ResolveEndpointsResponse{},
			expectedErr: errors.New("endpoint 'myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net' has no valid IP addresses"),
		},
		{
			name: "endpoint_not_exists",
			protoMsg: &pb.TRspResolveEndpoints{
				Timestamp:     uint64(now.Unix()),
				ResolveStatus: pb.EResolveStatus_NOT_EXISTS,
				WatchToken:    "0",
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				Ruid:          "ololo",
			},
			expected: ResolveEndpointsResponse{
				Timestamp:     uint64(now.Unix()),
				ResolveStatus: StatusEndpointNotExists,
				WatchToken:    "0",
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				RUID:          "ololo",
			},
			expectedErr: nil,
		},
		{
			name: "full_response",
			protoMsg: &pb.TRspResolveEndpoints{
				Timestamp: uint64(now.Unix()),
				EndpointSet: &pb.TEndpointSet{
					EndpointSetId: "trololo",
					Endpoints: []*pb.TEndpoint{
						{
							Id:                   "shimba-boomba",
							Protocol:             "TCP",
							Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
							Ip6Address:           "2a02:6b8:c08:afa8:10d:bd77:7f89:0",
							Port:                 8080,
							LabelSelectorResults: []string{"looken-tooken"},
							Ready:                true,
						},
					},
				},
				ResolveStatus: pb.EResolveStatus_OK,
				WatchToken:    "0",
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				Ruid:          "ololo",
			},
			expected: ResolveEndpointsResponse{
				Timestamp: uint64(now.Unix()),
				EndpointSet: &EndpointSet{
					ID: "trololo",
					Endpoints: []*Endpoint{
						{
							ID:       "shimba-boomba",
							Protocol: "TCP",
							FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
							IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
							Port:     8080,
							Labels:   []string{"looken-tooken"},
							Ready:    true,
						},
					},
				},
				ResolveStatus: StatusEndpointOK,
				WatchToken:    "0",
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				RUID:          "ololo",
			},
			expectedErr: nil,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var actual ResolveEndpointsResponse
			err := actual.UnmarshalProto(tc.protoMsg)

			if tc.expectedErr == nil {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tc.expectedErr.Error())
			}

			assert.Equal(t, tc.expected, actual)
		})
	}
}

func TestEndpointSet_UnmarshalProto(t *testing.T) {
	testCases := []struct {
		name        string
		protoMsg    *pb.TEndpointSet
		expected    EndpointSet
		expectedErr error
	}{
		{
			name: "error_no_ips",
			protoMsg: &pb.TEndpointSet{
				EndpointSetId: "trololo",
				Endpoints: []*pb.TEndpoint{
					{
						Id:                   "shimba-boomba",
						Protocol:             "TCP",
						Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
						Port:                 8080,
						LabelSelectorResults: []string{"looken-tooken"},
						Ready:                true,
					},
				},
			},
			expected:    EndpointSet{},
			expectedErr: errors.New("endpoint 'myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net' has no valid IP addresses"),
		},
		{
			name: "full_response",
			protoMsg: &pb.TEndpointSet{
				EndpointSetId: "trololo",
				Endpoints: []*pb.TEndpoint{
					{
						Id:                   "shimba-boomba",
						Protocol:             "TCP",
						Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
						Ip6Address:           "2a02:6b8:c08:afa8:10d:bd77:7f89:0",
						Port:                 8080,
						LabelSelectorResults: []string{"looken-tooken"},
						Ready:                true,
					},
				},
			},
			expected: EndpointSet{
				ID: "trololo",
				Endpoints: []*Endpoint{
					{
						ID:       "shimba-boomba",
						Protocol: "TCP",
						FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
						IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
						Port:     8080,
						Labels:   []string{"looken-tooken"},
						Ready:    true,
					},
				},
			},
			expectedErr: nil,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var actual EndpointSet
			err := actual.UnmarshalProto(tc.protoMsg)

			if tc.expectedErr == nil {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tc.expectedErr.Error())
			}

			assert.Equal(t, tc.expected, actual)
		})
	}
}

func TestEndpoint_UnmarshalProto(t *testing.T) {
	testCases := []struct {
		name        string
		protoMsg    *pb.TEndpoint
		expected    Endpoint
		expectedErr error
	}{
		{
			name: "error_no_ips",
			protoMsg: &pb.TEndpoint{

				Id:                   "shimba-boomba",
				Protocol:             "TCP",
				Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
				Port:                 8080,
				LabelSelectorResults: []string{"looken-tooken"},
				Ready:                true,
			},
			expected:    Endpoint{},
			expectedErr: errors.New("endpoint 'myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net' has no valid IP addresses"),
		},
		{
			name: "full_response",
			protoMsg: &pb.TEndpoint{
				Id:                   "shimba-boomba",
				Protocol:             "TCP",
				Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
				Ip6Address:           "2a02:6b8:c08:afa8:10d:bd77:7f89:0",
				Port:                 8080,
				LabelSelectorResults: []string{"looken-tooken"},
				Ready:                true,
			},
			expected: Endpoint{
				ID:       "shimba-boomba",
				Protocol: "TCP",
				FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
				IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
				Port:     8080,
				Labels:   []string{"looken-tooken"},
				Ready:    true,
			},
			expectedErr: nil,
		},
		{
			name: "full_response_not_ready",
			protoMsg: &pb.TEndpoint{
				Id:                   "shimba-boomba",
				Protocol:             "TCP",
				Fqdn:                 "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
				Ip6Address:           "2a02:6b8:c08:afa8:10d:bd77:7f89:0",
				Port:                 8080,
				LabelSelectorResults: []string{"looken-tooken"},
				Ready:                false,
			},
			expected: Endpoint{
				ID:       "shimba-boomba",
				Protocol: "TCP",
				FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
				IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
				Port:     8080,
				Labels:   []string{"looken-tooken"},
				Ready:    false,
			},
			expectedErr: nil,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var actual Endpoint
			err := actual.UnmarshalProto(tc.protoMsg)

			if tc.expectedErr == nil {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tc.expectedErr.Error())
			}

			assert.Equal(t, tc.expected, actual)
		})
	}
}

func TestResolvePodsResponse_UnmarshalProto(t *testing.T) {
	now := time.Now()

	testCases := []struct {
		name        string
		protoMsg    *pb.TRspResolvePods
		expected    ResolvePodsResponse
		expectedErr error
	}{
		{
			name: "pod_not_exists",
			protoMsg: &pb.TRspResolvePods{
				Timestamp:     uint64(now.Unix()),
				ResolveStatus: pb.EResolveStatus_NOT_EXISTS,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				Ruid:          "ololo",
			},
			expected: ResolvePodsResponse{
				Timestamp:     uint64(now.Unix()),
				ResolveStatus: StatusPodNotExists,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				RUID:          "ololo",
			},
			expectedErr: nil,
		},
		{
			name: "full_response",
			protoMsg: &pb.TRspResolvePods{
				Timestamp: uint64(now.Unix()),
				PodSet: &pb.TPodSet{
					PodSetId: "sas-yt-seneca-sas-nodes-over-yp",
					Pods: []*pb.TPod{
						{
							Id:     "sas2-9558-node-seneca-sas",
							NodeId: "sas2-9558.search.yandex.net",
							Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
								{
									Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
									VlanId:         "fastbone",
									PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
									TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
									InternetAddress: &pb.TPod_TIP6AddressAllocation_TInternetAddress{
										Id:         "Id",
										Ip4Address: "Ip4Address",
									},
									VirtualServices: []*pb.TPod_TIP6AddressAllocation_TVirtualService{
										{
											Ip6Addresses: []string{"Ip6Addresses_0", "Ip6Addresses_1"},
											Ip4Addresses: []string{"Ip4Addresses_0", "Ip4Addresses_1"},
										},
									},
								},
							},
							Ip6SubnetAllocations: []*pb.TPod_TIP6SubnetAllocation{
								{
									Subnet: "Subnet",
									VlanId: "fastbone",
								},
							},
							Dns: &pb.TPod_TDns{
								PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							},
							IssConfSummaries: map[string]*pb.TPod_TIssConfSummary{
								"sas2-9558-node-seneca-sas-1642488699987": {
									TargetState: "PREPARED",
								},
								"sas2-9558-node-seneca-sas-1642499382179": {
									TargetState: "ACTIVE",
								},
							},
						},
					},
				},
				ResolveStatus: pb.EResolveStatus_OK,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				Ruid:          "ololo",
			},
			expected: ResolvePodsResponse{
				Timestamp: uint64(now.Unix()),
				PodSet: &PodSet{
					ID: "sas-yt-seneca-sas-nodes-over-yp",
					Pods: []*Pod{
						{
							ID:     "sas2-9558-node-seneca-sas",
							NodeID: "sas2-9558.search.yandex.net",
							IP6AddressAllocations: []IP6AddressAllocation{
								{
									Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
									VLANID:         "fastbone",
									PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
									TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
									InternetAddress: &InternetAddress{
										ID:         "Id",
										IP4Address: "Ip4Address",
									},
									VirtualServices: []VirtualService{
										{
											IP6Addresses: []string{"Ip6Addresses_0", "Ip6Addresses_1"},
											IP4Addresses: []string{"Ip4Addresses_0", "Ip4Addresses_1"},
										},
									},
								},
							},
							IP6SubnetAllocations: []IP6SubnetAllocation{
								{
									Subnet: "Subnet",
									VLANID: "fastbone",
								},
							},
							DNS: &DNS{
								PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							},
							IssConfSummaries: map[string]IssConfSummary{
								"sas2-9558-node-seneca-sas-1642488699987": {
									TargetState: "PREPARED",
								},
								"sas2-9558-node-seneca-sas-1642499382179": {
									TargetState: "ACTIVE",
								},
							},
						},
					},
				},
				ResolveStatus: StatusPodOK,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				RUID:          "ololo",
			},
			expectedErr: nil,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var actual ResolvePodsResponse
			err := actual.UnmarshalProto(tc.protoMsg)

			if tc.expectedErr == nil {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tc.expectedErr.Error())
			}

			assert.Equal(t, tc.expected, actual)
		})
	}
}

func TestResolveNodeResponse_UnmarshalProto(t *testing.T) {
	now := time.Now()

	testCases := []struct {
		name        string
		protoMsg    *pb.TRspResolveNode
		expected    ResolveNodeResponse
		expectedErr error
	}{
		{
			name: "node_not_exists",
			protoMsg: &pb.TRspResolveNode{
				Timestamp:     uint64(now.Unix()),
				ResolveStatus: pb.EResolveStatus_NOT_EXISTS,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				Ruid:          "ololo",
			},
			expected: ResolveNodeResponse{
				Pods:          []*Pod{},
				Timestamp:     uint64(now.Unix()),
				ResolveStatus: StatusPodNotExists,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				RUID:          "ololo",
			},
			expectedErr: nil,
		},
		{
			name: "full_response",
			protoMsg: &pb.TRspResolveNode{
				Timestamp: uint64(now.Unix()),
				Pods: []*pb.TPod{
					{
						Id:     "sas2-9558-node-seneca-sas",
						NodeId: "sas2-9558.search.yandex.net",
						Ip6AddressAllocations: []*pb.TPod_TIP6AddressAllocation{
							{
								Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
								VlanId:         "fastbone",
								PersistentFqdn: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFqdn:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								InternetAddress: &pb.TPod_TIP6AddressAllocation_TInternetAddress{
									Id:         "Id",
									Ip4Address: "Ip4Address",
								},
								VirtualServices: []*pb.TPod_TIP6AddressAllocation_TVirtualService{
									{
										Ip6Addresses: []string{"Ip6Addresses_0", "Ip6Addresses_1"},
										Ip4Addresses: []string{"Ip4Addresses_0", "Ip4Addresses_1"},
									},
								},
							},
						},
						Ip6SubnetAllocations: []*pb.TPod_TIP6SubnetAllocation{
							{
								Subnet: "Subnet",
								VlanId: "fastbone",
							},
						},
						Dns: &pb.TPod_TDns{
							PersistentFqdn: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFqdn:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
				},
				ResolveStatus: pb.EResolveStatus_OK,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				Ruid:          "ololo",
			},
			expected: ResolveNodeResponse{
				Timestamp: uint64(now.Unix()),
				Pods: []*Pod{
					{
						ID:     "sas2-9558-node-seneca-sas",
						NodeID: "sas2-9558.search.yandex.net",
						IP6AddressAllocations: []IP6AddressAllocation{
							{
								Address:        "2a02:6b8:fc17:78d:10d:adb5:ccb9:0",
								VLANID:         "fastbone",
								PersistentFQDN: "fb-sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								TransientFQDN:  "fb-sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
								InternetAddress: &InternetAddress{
									ID:         "Id",
									IP4Address: "Ip4Address",
								},
								VirtualServices: []VirtualService{
									{
										IP6Addresses: []string{"Ip6Addresses_0", "Ip6Addresses_1"},
										IP4Addresses: []string{"Ip4Addresses_0", "Ip4Addresses_1"},
									},
								},
							},
						},
						IP6SubnetAllocations: []IP6SubnetAllocation{
							{
								Subnet: "Subnet",
								VLANID: "fastbone",
							},
						},
						DNS: &DNS{
							PersistentFQDN: "sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
							TransientFQDN:  "sas2-9558-1.sas2-9558-node-seneca-sas.sas.yp-c.yandex.net",
						},
					},
				},
				ResolveStatus: StatusPodOK,
				Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
				RUID:          "ololo",
			},
			expectedErr: nil,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var actual ResolveNodeResponse
			err := actual.UnmarshalProto(tc.protoMsg)

			if tc.expectedErr == nil {
				assert.NoError(t, err)
			} else {
				assert.EqualError(t, err, tc.expectedErr.Error())
			}

			assert.Equal(t, tc.expected, actual)
		})
	}
}
