package grpcresolver

import (
	"context"
	"fmt"
	"sync"

	"google.golang.org/grpc/grpclog"
	"google.golang.org/grpc/resolver"

	ypResolver "a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver/grpcresolver"
)

var _ resolver.Builder = (*Builder)(nil)

type Builder struct {
	initOnce sync.Once
	initErr  error
	closeRes bool
	resolver ypResolver.Resolver
	t        target
}

var logger = grpclog.Component("ypsd")

func init() {
	resolver.Register(NewBuilder())
}

func NewBuilder(opts ...BuilderOption) *Builder {
	b := &Builder{
		t: target{
			freq:    defaultFreq,
			timeout: defaultResolverTimeout,
			resRate: defaultResolverRate,
		},
	}

	for _, opt := range opts {
		opt(b)
	}

	return b
}

func (b *Builder) Scheme() string {
	return Scheme
}

func (b *Builder) Build(target resolver.Target, cc resolver.ClientConn, _ resolver.BuildOptions) (resolver.Resolver, error) {
	logger.Infof("creating resolver for target: %+v", target)

	t, err := b.t.Parse(&target.URL)
	if err != nil {
		return nil, fmt.Errorf("invalid target: %w", err)
	}

	b.initOnce.Do(func() {
		if b.resolver != nil {
			return
		}

		b.resolver, b.initErr = grpcresolver.New(grpcresolver.WithClientName(defaultClientName))
		if b.initErr == nil {
			b.closeRes = true
		}
	})

	if b.initErr != nil {
		return nil, fmt.Errorf("init failed: %w", b.initErr)
	}

	ctx, cancel := context.WithCancel(context.Background())
	yr := &Resolver{
		t:        t,
		ctx:      ctx,
		cancel:   cancel,
		resolver: b.resolver,
		cc:       cc,
		rn:       make(chan struct{}, 1),
	}

	yr.wg.Add(1)

	go yr.watcher()
	return yr, nil
}

func (b *Builder) Close() error {
	if b.closeRes {
		return b.resolver.Close()
	}
	return nil
}
