package grpcresolver_test

import (
	"context"
	"fmt"
	"net"
	"net/url"
	"strings"
	"sync"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"google.golang.org/grpc/resolver"

	ypResolver "a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	ypMockResolver "a.yandex-team.ru/infra/yp_service_discovery/golang/resolver/mockresolver"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/wrapper/grpcresolver"
)

type testAddress struct {
	Addr       string
	ID         string
	Cluster    string
	ServerName string
	Labels     []string
}

type testClientConn struct {
	resolver.ClientConn // For unimplemented functions
	target              string
	m1                  sync.Mutex
	state               resolver.State
	updateStateCalls    int
	errChan             chan error
}

func (t *testClientConn) UpdateState(s resolver.State) error {
	t.m1.Lock()
	defer t.m1.Unlock()
	t.state = s
	t.updateStateCalls++
	return nil
}

func (t *testClientConn) getState() (resolver.State, int) {
	t.m1.Lock()
	defer t.m1.Unlock()
	return t.state, t.updateStateCalls
}

func (t *testClientConn) reset() {
	t.m1.Lock()
	defer t.m1.Unlock()
	t.updateStateCalls = 0
	t.state = resolver.State{}
}

func parseTarget(t *testing.T, target string) resolver.Target {
	u, err := url.Parse(target)
	require.NoErrorf(t, err, "parse target: %s", target)

	endpoint := u.Path
	if endpoint == "" {
		endpoint = u.Opaque
	}
	endpoint = strings.TrimPrefix(endpoint, "/")
	return resolver.Target{
		Scheme:    u.Scheme,
		Authority: u.Host,
		Endpoint:  endpoint,
		URL:       *u,
	}
}

func parseAddresses(addrs []resolver.Address) []testAddress {
	out := make([]testAddress, len(addrs))
	for i, addr := range addrs {
		out[i] = testAddress{
			Addr:       addr.Addr,
			ServerName: addr.ServerName,
			ID:         grpcresolver.AddressID(addr),
			Cluster:    grpcresolver.AddressCluster(addr),
			Labels:     grpcresolver.AddressLabels(addr),
		}
	}
	return out
}

func TestResolver(t *testing.T) {
	cases := []struct {
		name      string
		ypResolve func(ctx context.Context, cluster, endpointSet string) (*ypResolver.ResolveEndpointsResponse, error)
		targetB   *grpcresolver.TargetBuilder
		addresses []testAddress
	}{
		{
			name: "single",
			ypResolve: func(ctx context.Context, cluster, endpointSet string) (*ypResolver.ResolveEndpointsResponse, error) {
				return &ypResolver.ResolveEndpointsResponse{
					Timestamp: 1,
					EndpointSet: &ypResolver.EndpointSet{
						ID: "single",
						Endpoints: []*ypResolver.Endpoint{
							{
								ID:       "single-1",
								Protocol: "TCP",
								FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
								IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
								Port:     8080,
								Labels:   []string{"my-label"},
								Ready:    true,
							},
						},
					},
					ResolveStatus: ypResolver.StatusEndpointOK,
					WatchToken:    "0",
					Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
					RUID:          "single",
				}, nil
			},
			targetB: grpcresolver.NewTargetBuilder("single", "kek"),
			addresses: []testAddress{
				{
					Addr:    "[2a02:6b8:c08:afa8:10d:bd77:7f89:0]:8080",
					ID:      "single-1",
					Cluster: "kek",
					Labels:  []string{"my-label"},
				},
			},
		},
		{
			name: "single-name",
			ypResolve: func(ctx context.Context, cluster, endpointSet string) (*ypResolver.ResolveEndpointsResponse, error) {
				return &ypResolver.ResolveEndpointsResponse{
					Timestamp: 1,
					EndpointSet: &ypResolver.EndpointSet{
						ID: "single-name",
						Endpoints: []*ypResolver.Endpoint{
							{
								ID:       "single-name-1",
								Protocol: "TCP",
								FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
								IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
								Port:     8080,
								Labels:   []string{"my-label"},
								Ready:    true,
							},
						},
					},
					ResolveStatus: ypResolver.StatusEndpointOK,
					WatchToken:    "0",
					Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
					RUID:          "single-name",
				}, nil
			},
			targetB: grpcresolver.NewTargetBuilder("single-name", "kek").WithServerName("named"),
			addresses: []testAddress{
				{
					Addr:       "[2a02:6b8:c08:afa8:10d:bd77:7f89:0]:8080",
					ID:         "single-name-1",
					ServerName: "named",
					Cluster:    "kek",
					Labels:     []string{"my-label"},
				},
			},
		},
		{
			name: "filter-ready",
			ypResolve: func(ctx context.Context, cluster, endpointSet string) (*ypResolver.ResolveEndpointsResponse, error) {
				return &ypResolver.ResolveEndpointsResponse{
					Timestamp: 1,
					EndpointSet: &ypResolver.EndpointSet{
						ID: "filter-ready",
						Endpoints: []*ypResolver.Endpoint{
							{
								ID:       "ready",
								Protocol: "TCP",
								FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
								IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
								Port:     8080,
								Labels:   []string{"label-ready"},
								Ready:    true,
							},
							{
								ID:       "not-ready",
								Protocol: "TCP",
								FQDN:     "myt1-1717-msk-yp-service-discovery-20076.gencfg-c.yandex.net",
								IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:1"),
								Port:     8080,
								Labels:   []string{"label-not-ready"},
							},
						},
					},
					ResolveStatus: ypResolver.StatusEndpointOK,
					WatchToken:    "0",
					Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
					RUID:          "filter-ready",
				}, nil
			},
			targetB: grpcresolver.NewTargetBuilder("filter-ready", "kek"),
			addresses: []testAddress{
				{
					Addr:    "[2a02:6b8:c08:afa8:10d:bd77:7f89:0]:8080",
					ID:      "ready",
					Cluster: "kek",
					Labels:  []string{"label-ready"},
				},
			},
		},
		{
			name: "ips-priority",
			ypResolve: func(ctx context.Context, cluster, endpointSet string) (*ypResolver.ResolveEndpointsResponse, error) {
				return &ypResolver.ResolveEndpointsResponse{
					Timestamp: 1,
					EndpointSet: &ypResolver.EndpointSet{
						ID: "ips-priority",
						Endpoints: []*ypResolver.Endpoint{
							{
								ID:       "both",
								Protocol: "TCP",
								FQDN:     "both.yandex.net",
								IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
								IPv4:     net.ParseIP("1.1.1.1"),
								Port:     8080,
								Ready:    true,
							},
							{
								ID:       "ipv6",
								Protocol: "TCP",
								FQDN:     "ipv6.yandex.net",
								IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:1"),
								Port:     8080,
								Ready:    true,
							},
							{
								ID:       "ipv4",
								Protocol: "TCP",
								FQDN:     "ipv4.yandex.net",
								IPv4:     net.ParseIP("1.1.1.1"),
								Port:     8080,
								Ready:    true,
							},
							{
								ID:       "ipv6-unspecified",
								Protocol: "TCP",
								FQDN:     "ipv6-unspecified.yandex.net",
								IPv6:     net.IPv6unspecified,
								IPv4:     net.ParseIP("1.1.1.1"),
								Port:     8080,
								Ready:    true,
							},
							{
								ID:       "no-addr",
								Protocol: "TCP",
								FQDN:     "ipv6-unspecified.yandex.net",
								Port:     8080,
								Ready:    true,
							},
							{
								ID:       "unspecified",
								Protocol: "TCP",
								FQDN:     "ipv4-unspecified.yandex.net",
								IPv6:     net.IPv6unspecified,
								IPv4:     net.IPv4zero,
								Port:     8080,
								Ready:    true,
							},
						},
					},
					ResolveStatus: ypResolver.StatusEndpointOK,
					WatchToken:    "0",
					Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
					RUID:          "ips-priority",
				}, nil
			},
			targetB: grpcresolver.NewTargetBuilder("ips", "kek"),
			addresses: []testAddress{
				{
					Addr:    "[2a02:6b8:c08:afa8:10d:bd77:7f89:0]:8080",
					ID:      "both",
					Cluster: "kek",
				},
				{
					Addr:    "[2a02:6b8:c08:afa8:10d:bd77:7f89:1]:8080",
					ID:      "ipv6",
					Cluster: "kek",
				},
				{
					Addr:    "1.1.1.1:8080",
					ID:      "ipv4",
					Cluster: "kek",
				},
				{
					Addr:    "1.1.1.1:8080",
					ID:      "ipv6-unspecified",
					Cluster: "kek",
				},
			},
		},
		{
			name: "multi-cluster",
			ypResolve: func(ctx context.Context, cluster, endpointSet string) (*ypResolver.ResolveEndpointsResponse, error) {
				switch cluster {
				case "first":
					return &ypResolver.ResolveEndpointsResponse{
						Timestamp: 1,
						EndpointSet: &ypResolver.EndpointSet{
							ID: "multi-cluster",
							Endpoints: []*ypResolver.Endpoint{
								{
									ID:       "first-1",
									Protocol: "TCP",
									FQDN:     "myt1-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
									IPv6:     net.ParseIP("2a01:6b8:c08:afa8:10d:bd77:7f89:0"),
									Port:     8080,
									Ready:    true,
									Labels:   []string{"first-label"},
								},
								{
									ID:       "first-2",
									Protocol: "TCP",
									FQDN:     "myt1-1718-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
									IPv6:     net.ParseIP("2a01:6b8:c08:afa8:10d:bd77:7f89:1"),
									Port:     8080,
									Ready:    true,
									Labels:   []string{"first-label"},
								},
							},
						},
						ResolveStatus: ypResolver.StatusEndpointOK,
						WatchToken:    "0",
						Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
						RUID:          "multi-cluster",
					}, nil
				case "second":
					return &ypResolver.ResolveEndpointsResponse{
						Timestamp: 1,
						EndpointSet: &ypResolver.EndpointSet{
							ID: "multi-cluster",
							Endpoints: []*ypResolver.Endpoint{
								{
									ID:       "second-1",
									Protocol: "TCP",
									FQDN:     "myt2-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
									IPv6:     net.ParseIP("2a02:6b8:c08:afa8:10d:bd77:7f89:0"),
									Port:     8081,
									Ready:    true,
									Labels:   []string{"second-label"},
								},
							},
						},
						ResolveStatus: ypResolver.StatusEndpointOK,
						WatchToken:    "0",
						Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
						RUID:          "multi-cluster",
					}, nil
				case "third":
					return &ypResolver.ResolveEndpointsResponse{
						Timestamp: 1,
						EndpointSet: &ypResolver.EndpointSet{
							ID: "multi-cluster",
							Endpoints: []*ypResolver.Endpoint{
								{
									ID:       "third-1",
									Protocol: "TCP",
									FQDN:     "myt3-1717-msk-yp-service-discovery-20075.gencfg-c.yandex.net",
									IPv6:     net.ParseIP("2a03:6b8:c08:afa8:10d:bd77:7f89:0"),
									Port:     8081,
									Ready:    true,
									Labels:   []string{"third-label"},
								},
							},
						},
						ResolveStatus: ypResolver.StatusEndpointNotExists,
						WatchToken:    "0",
						Host:          "sas3-1449-8be-sas-yp-service-d-419-22443.gencfg-c.yandex.net",
						RUID:          "multi-cluster",
					}, nil
				default:
					return nil, fmt.Errorf("unexpected cluster: %s", cluster)
				}
			},
			targetB: grpcresolver.NewTargetBuilder("multi", "first", "second", "third"),
			addresses: []testAddress{
				{
					Addr:    "[2a01:6b8:c08:afa8:10d:bd77:7f89:0]:8080",
					ID:      "first-1",
					Cluster: "first",
					Labels:  []string{"first-label"},
				},
				{
					Addr:    "[2a01:6b8:c08:afa8:10d:bd77:7f89:1]:8080",
					ID:      "first-2",
					Cluster: "first",
					Labels:  []string{"first-label"},
				},
				{
					Addr:    "[2a02:6b8:c08:afa8:10d:bd77:7f89:0]:8081",
					ID:      "second-1",
					Cluster: "second",
					Labels:  []string{"second-label"},
				},
			},
		},
	}

	for _, tc := range cases {
		t.Run(tc.name, func(t *testing.T) {
			r := ypMockResolver.Resolver{MockResolveEndpoints: tc.ypResolve}
			b := grpcresolver.NewBuilder(grpcresolver.WithYpResolver(r))
			defer func() {
				err := b.Close()
				require.NoError(t, err, "close failed")
			}()

			target := tc.targetB.Build()
			cc := &testClientConn{target: target}
			grpcResolver, err := b.Build(parseTarget(t, target), cc, resolver.BuildOptions{})
			require.NoError(t, err)
			defer grpcResolver.Close()

			var state resolver.State
			var cnt int
			for i := 0; i < 2000; i++ {
				state, cnt = cc.getState()
				if cnt > 0 {
					break
				}
				time.Sleep(time.Millisecond)
			}
			require.NotZero(t, cnt, "UpdateState not called after 2s; aborting")
			require.EqualValues(t, tc.addresses, parseAddresses(state.Addresses))
			for _, addr := range state.Addresses {
				assert.Truef(t, addr.Equal(addr), "addr %q is not equal to itself", addr.Addr)
			}
		})
	}
}

func TestTargetErrors(t *testing.T) {
	cases := []struct {
		target resolver.Target
		err    string
	}{
		{
			target: resolver.Target{
				Scheme:    grpcresolver.Scheme,
				Authority: "no_url",
				Endpoint:  "something",
			},
			err: "invalid target: no clusters provided",
		},
		{
			target: resolver.Target{
				URL: url.URL{
					Scheme: grpcresolver.Scheme,
					Path:   "/endpoint_only",
				},
			},
			err: "invalid target: no clusters provided",
		},
		{
			target: resolver.Target{
				URL: url.URL{
					Scheme: grpcresolver.Scheme,
					Opaque: "/endpoint_only",
				},
			},
			err: "invalid target: no clusters provided",
		},
		{
			target: resolver.Target{
				URL: url.URL{
					Scheme: grpcresolver.Scheme,
					Host:   "authority_only",
				},
			},
			err: "invalid target: no endpointSet set provided",
		},
		{
			target: resolver.Target{
				URL: url.URL{
					Scheme: grpcresolver.Scheme,
					Host:   "authority_only",
					Path:   "/",
					Opaque: "/",
				},
			},
			err: "invalid target: no endpointSet set provided",
		},
	}

	b := grpcresolver.NewBuilder(grpcresolver.WithYpResolver(ypMockResolver.Resolver{}))
	defer func() { _ = b.Close() }()

	for _, tc := range cases {
		t.Run(tc.target.URL.String(), func(t *testing.T) {
			r, err := b.Build(tc.target, &testClientConn{}, resolver.BuildOptions{})
			require.EqualError(t, err, tc.err)
			require.Nil(t, r)
		})
	}
}
