package grpcresolver

import (
	"errors"
	"fmt"
	"net/url"
	"strconv"
	"strings"
	"time"
)

const (
	refreshFreqKey     = "refresh_frequency"
	resolverTimeoutKey = "resolver_timeout"
	reResRateKey       = "re_resolution_rate"
	serverNameKey      = "server_name"
	allowEmptyKey      = "allow_empty"
)

type TargetBuilder struct {
	u url.URL
	q url.Values
}

func NewTargetBuilder(endpointSet string, clusters ...string) *TargetBuilder {
	return &TargetBuilder{
		u: url.URL{
			Scheme: Scheme,
			Host:   strings.Join(clusters, "."),
			Path:   "/" + endpointSet,
		},
		q: url.Values{},
	}
}

// WithRefreshFrequency sets resolution frequency.
//
// Default: 1 minute
func (t *TargetBuilder) WithRefreshFrequency(freq time.Duration) *TargetBuilder {
	t.q.Set(refreshFreqKey, freq.String())
	return t
}

// WithResolverTimeout sets timeout for each request to YP.SD.
//
// Default: 1 second
func (t *TargetBuilder) WithResolverTimeout(timeout time.Duration) *TargetBuilder {
	t.q.Set(resolverTimeoutKey, timeout.String())
	return t
}

// WithReResolutionRate sets re-resolution rate limits (i.e. how often we can request YP SD).
// This prevents excessive re-resolution and allow to configure exponential backoff re-resolution in case of trouble.
//
// Default: 5 seconds
func (t *TargetBuilder) WithReResolutionRate(rate time.Duration) *TargetBuilder {
	t.q.Set(reResRateKey, rate.String())
	return t
}

// WithServerName sets server name that used as the transport certification authority for the address.
func (t *TargetBuilder) WithServerName(name string) *TargetBuilder {
	t.q.Set(serverNameKey, name)
	return t
}

// WithAllowEmptyAddressList allows resolver to retrun empty adress list from YP SD request,
// which can be returned, in some cases (see README.md)
//
// Default: false
func (t *TargetBuilder) WithAllowEmptyAddressList() *TargetBuilder {
	t.q.Set(allowEmptyKey, "true")
	return t
}

func (t *TargetBuilder) Build() string {
	t.u.RawQuery = t.q.Encode()
	return t.u.String()
}

func (t *TargetBuilder) String() string {
	return t.Build()
}

type target struct {
	endpointSet string
	clusters    []string
	freq        time.Duration
	timeout     time.Duration
	resRate     time.Duration
	serverName  string
	allowEmpty  bool
}

func (t target) Parse(u *url.URL) (target, error) {
	if authority := u.Host; authority != "" {
		t.clusters = strings.Split(authority, ".")
	}

	if len(t.clusters) == 0 {
		return target{}, errors.New("no clusters provided")
	}

	t.endpointSet = u.Path
	if t.endpointSet == "" {
		t.endpointSet = u.Opaque
	}
	t.endpointSet = strings.TrimPrefix(t.endpointSet, "/")

	if t.endpointSet == "" {
		return target{}, errors.New("no endpointSet set provided")
	}

	parsers := map[string]func(string) error{
		refreshFreqKey: func(s string) (err error) {
			t.freq, err = time.ParseDuration(s)
			return
		},
		resolverTimeoutKey: func(s string) (err error) {
			t.timeout, err = time.ParseDuration(s)
			return
		},
		reResRateKey: func(s string) (err error) {
			t.resRate, err = time.ParseDuration(s)
			return
		},
		serverNameKey: func(s string) error {
			t.serverName = s
			return nil
		},
		allowEmptyKey: func(s string) (err error) {
			t.allowEmpty, err = strconv.ParseBool(s)
			return
		},
	}
	query := u.Query()
	for k, p := range parsers {
		val := query.Get(k)
		if val == "" {
			continue
		}

		if err := p(val); err != nil {
			return target{}, fmt.Errorf("invalid %q parameter: %w", k, err)
		}
	}

	return t, nil
}
