package grpcresolver

import (
	"net/url"
	"testing"
	"time"

	"github.com/stretchr/testify/require"
)

func mergeTargets(a, b target) target {
	if b.endpointSet != "" {
		a.endpointSet = b.endpointSet
	}

	if len(b.clusters) > 0 {
		a.clusters = b.clusters
	}

	if b.freq != 0 {
		a.freq = b.freq
	}

	if b.timeout != 0 {
		a.timeout = b.timeout
	}

	if b.resRate != 0 {
		a.resRate = b.resRate
	}

	if b.serverName != "" {
		a.serverName = b.serverName
	}

	if b.allowEmpty {
		a.allowEmpty = b.allowEmpty
	}

	return a
}

func TestTargetParse(t *testing.T) {
	defaultTarget := target{
		freq:       time.Minute,
		resRate:    time.Hour,
		timeout:    time.Second,
		allowEmpty: false,
	}

	cases := []struct {
		u   string
		t   target
		err string
	}{
		{
			u:   "grpcypsd://vla",
			err: "no endpointSet set provided",
		},
		{
			u:   "grpcypsd:///set",
			err: "no clusters provided",
		},
		{
			u: "grpcypsd://vla/set",
			t: mergeTargets(defaultTarget, target{
				endpointSet: "set",
				clusters:    []string{"vla"},
			}),
		},
		{
			u: "grpcypsd://man.vla/endpointSet?refresh_frequency=2m&resolver_timeout=2s&re_resolution_rate=20s&allow_empty=true&server_name=lol",
			t: mergeTargets(defaultTarget, target{
				endpointSet: "endpointSet",
				clusters:    []string{"man", "vla"},
				serverName:  "lol",
				freq:        2 * time.Minute,
				timeout:     2 * time.Second,
				resRate:     20 * time.Second,
				allowEmpty:  true,
			}),
		},
	}

	for _, tc := range cases {
		t.Run(tc.u, func(t *testing.T) {
			u, err := url.Parse(tc.u)
			require.NoError(t, err)

			actual, err := defaultTarget.Parse(u)
			if tc.err != "" {
				require.EqualError(t, err, tc.err)
				return
			}

			require.NoError(t, err)
			require.EqualValues(t, tc.t, actual)
		})
	}
}

func TestTargetBuild(t *testing.T) {
	defaultTarget := target{
		freq:       time.Minute,
		resRate:    time.Hour,
		timeout:    time.Second,
		allowEmpty: false,
	}

	cases := []struct {
		name string
		b    *TargetBuilder
		t    target
	}{
		{
			name: "minimal",
			b:    NewTargetBuilder("endpoint_set", "vla"),
			t: mergeTargets(defaultTarget, target{
				endpointSet: "endpoint_set",
				clusters:    []string{"vla"},
			}),
		},
		{
			name: "full",
			b: NewTargetBuilder("endpointSet", "vla", "man").
				WithRefreshFrequency(100 * time.Second).
				WithResolverTimeout(2 * time.Second).
				WithReResolutionRate(1 * time.Minute).
				WithServerName("kek").
				WithAllowEmptyAddressList(),
			t: mergeTargets(defaultTarget, target{
				endpointSet: "endpointSet",
				clusters:    []string{"vla", "man"},
				serverName:  "kek",
				freq:        100 * time.Second,
				timeout:     2 * time.Second,
				resRate:     time.Minute,
				allowEmpty:  true,
			}),
		},
	}

	for _, tc := range cases {
		t.Run(tc.name, func(t *testing.T) {
			u, err := url.Parse(tc.b.Build())
			require.NoError(t, err)

			actual, err := defaultTarget.Parse(u)
			require.NoError(t, err)
			require.EqualValues(t, tc.t, actual)
		})
	}
}
