package ru.yandex.yp.discovery;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import io.grpc.ClientInterceptor;

import ru.yandex.yp.discovery.impl.YpDiscoveryClientImpl;
import ru.yandex.yp.discovery.impl.YpDiscoveryClientMonitoringStub;

/**
 * Raw YP service discovery client builder.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YpDiscoveryClientBuilder {

    private final YpDiscoveryInstance instance;
    private final String clientName;
    private final YpCredentialsProvider credentialsProvider;
    private final List<ClientInterceptor> interceptors = new ArrayList<>();

    private String userAgent = "Java YP service discovery client";
    private YpDiscoveryClientMonitoring monitoring = new YpDiscoveryClientMonitoringStub();
    private Duration idleTimeout;
    private Duration keepAliveTime;
    private Duration keepAliveTimeout;
    private Boolean keepAliveWithoutCalls;
    private Integer maxInboundMessageSize;
    private Integer maxInboundMetadataSize;
    private Duration callTimeout;
    private boolean usePlaintext = true;

    /**
     * Create raw YP service discovery client builder.
     *
     * @param instance YP service discovery instance to connect to.
     * @param clientName Default client name.
     * @param credentialsProvider Credentials provider.
     */
    public YpDiscoveryClientBuilder(YpDiscoveryInstance instance, String clientName, YpCredentialsProvider credentialsProvider) {
        this.instance = instance;
        this.clientName = clientName;
        this.credentialsProvider = credentialsProvider;
    }

    /**
     * Create raw YP service discovery client builder.
     *
     * @param instance YP service discovery instance to connect to.
     * @param clientName Default client name.
     */
    public YpDiscoveryClientBuilder(YpDiscoveryInstance instance, String clientName) {
        this.instance = instance;
        this.clientName = clientName;
        this.credentialsProvider = null;
    }

    /**
     * Get YP instance to connect to.
     *
     * @return YP instance to connect to.
     */
    public YpDiscoveryInstance getInstance() {
        return instance;
    }

    /**
     * Get default client name.
     *
     * @return Default client name.
     */
    public String getClientName() {
        return clientName;
    }

    /**
     * Get credentials provider.
     *
     * @return Credentials provider.
     */
    public Optional<YpCredentialsProvider> getCredentialsProvider() {
        return Optional.ofNullable(credentialsProvider);
    }

    /**
     * Get user agent.
     *
     * @return User agent.
     */
    public String getUserAgent() {
        return userAgent;
    }

    /**
     * Set user agent.
     *
     * @param userAgent User agent.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setUserAgent(String userAgent) {
        if (userAgent == null) {
            throw new IllegalArgumentException("User agent may not be null");
        }
        this.userAgent = userAgent;
        return this;
    }

    /**
     * Get client monitoring implementation.
     *
     * @return Client monitoring implementation.
     */
    public YpDiscoveryClientMonitoring getMonitoring() {
        return monitoring;
    }

    /**
     * Set client monitoring implementation.
     *
     * @param monitoring Client monitoring implementation.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setMonitoring(YpDiscoveryClientMonitoring monitoring) {
        if (monitoring == null) {
            throw new IllegalArgumentException("Monitoring may not be null");
        }
        this.monitoring = monitoring;
        return this;
    }

    /**
     * Get client interceptors.
     *
     * @return Client interceptors.
     */
    public List<ClientInterceptor> getInterceptors() {
        return Collections.unmodifiableList(interceptors);
    }

    /**
     * Add client interceptor.
     *
     * @param interceptor Client interceptor.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder addInterceptor(ClientInterceptor interceptor) {
        if (interceptor == null) {
            throw new IllegalArgumentException("Interceptor may not be null");
        }
        this.interceptors.add(interceptor);
        return this;
    }

    /**
     * Get idle timeout.
     *
     * @return Idle timeout.
     */
    public Optional<Duration> getIdleTimeout() {
        return Optional.ofNullable(idleTimeout);
    }

    /**
     * Set idle timeout.
     *
     * @param idleTimeout Idle timeout.
     * @param timeUnit Time unit.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setIdleTimeout(long idleTimeout, TimeUnit timeUnit) {
        this.idleTimeout = Duration.ofNanos(timeUnit.toNanos(idleTimeout));
        return this;
    }

    /**
     * Get keep alive time.
     *
     * @return Keep alive time.
     */
    public Optional<Duration> getKeepAliveTime() {
        return Optional.ofNullable(keepAliveTime);
    }

    /**
     * Set keep alive time.
     *
     * @param keepAliveTime Keep alive time.
     * @param timeUnit Time unit.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setKeepAliveTime(long keepAliveTime, TimeUnit timeUnit) {
        this.keepAliveTime = Duration.ofNanos(timeUnit.toNanos(keepAliveTime));
        return this;
    }

    /**
     * Get keep alive timeout.
     *
     * @return Keep alive timeout.
     */
    public Optional<Duration> getKeepAliveTimeout() {
        return Optional.ofNullable(keepAliveTimeout);
    }

    /**
     * Set keep alive timeout.
     *
     * @param keepAliveTimeout Keep alive timeout.
     * @param timeUnit Time unit.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setKeepAliveTimeout(long keepAliveTimeout, TimeUnit timeUnit) {
        this.keepAliveTimeout = Duration.ofNanos(timeUnit.toNanos(keepAliveTimeout));
        return this;
    }

    /**
     * Is keep alive without calls enabled?
     *
     * @return Is keep alive without calls enabled?
     */
    public Optional<Boolean> getKeepAliveWithoutCalls() {
        return Optional.ofNullable(keepAliveWithoutCalls);
    }

    /**
     * Set keep alive without calls.
     *
     * @param keepAliveWithoutCalls Is keep alive without calls enabled?
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setKeepAliveWithoutCalls(boolean keepAliveWithoutCalls) {
        this.keepAliveWithoutCalls = keepAliveWithoutCalls;
        return this;
    }

    /**
     * Get maximum inbound message size.
     *
     * @return Maximum inbound message size.
     */
    public Optional<Integer> getMaxInboundMessageSize() {
        return Optional.ofNullable(maxInboundMessageSize);
    }

    /**
     * Set maximum inbound message size.
     *
     * @param maxInboundMessageSize Maximum inbound message size.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setMaxInboundMessageSize(int maxInboundMessageSize) {
        this.maxInboundMessageSize = maxInboundMessageSize;
        return this;
    }

    /**
     * Get maximum inbound metadata size.
     *
     * @return Maximum inbound metadata size.
     */
    public Optional<Integer> getMaxInboundMetadataSize() {
        return Optional.ofNullable(maxInboundMetadataSize);
    }

    /**
     * Set maximum inbound metadata size.
     *
     * @param maxInboundMetadataSize Maximum inbound metadata size.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setMaxInboundMetadataSize(int maxInboundMetadataSize) {
        this.maxInboundMetadataSize = maxInboundMetadataSize;
        return this;
    }

    /**
     * Get per-call timeout value.
     * @return Deadline after setting value.
     */
    public Optional<Duration> getCallTimeout() {
        return Optional.ofNullable(callTimeout);
    }

    /**
     * Set per-call timeout value.
     * @param timeoutValue timeout value.
     * @param timeUnit Time unit for timeout value.
     * @return {@link YpDiscoveryClientBuilder}.
     */
    public YpDiscoveryClientBuilder setTimeout(long timeoutValue, TimeUnit timeUnit) {
        this.callTimeout = Duration.ofNanos(timeUnit.toNanos(timeoutValue));
        return this;
    }

    /**
     * True if client must skip establishing a TLS connection.
     * Defaults to false.
     * @return whether client must skip establishing a TLS connection.
     */
    public boolean usePlaintext() {
        return usePlaintext;
    }

    /**
     * Set whether client must skip establishing a TLS connection.
     * @param usePlaintext whether client must skip establishing a TLS connection.
     * @return {@link YpDiscoveryClientBuilder}
     */
    public YpDiscoveryClientBuilder setUsePlaintext(boolean usePlaintext) {
        this.usePlaintext = usePlaintext;
        return this;
    }

    /**
     * Build {@link YpDiscoveryClient}.
     * @return {@link YpDiscoveryClient}.
     */
    public YpDiscoveryClient build() {
        return new YpDiscoveryClientImpl(this);
    }

}
