package ru.yandex.yp.discovery.impl;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Supplier;

import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.yp.discovery.YpDiscoveryClientMonitoring;
import ru.yandex.yp.discovery.YpDiscoveryRequestType;

/**
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YpClientMonitoringUtils {

    private final static Logger logger = LoggerFactory.getLogger(YpClientMonitoringUtils.class);

    private YpClientMonitoringUtils() {
    }

    public static <T> CompletableFuture<T> measure(YpDiscoveryClientMonitoring monitoring, YpDiscoveryRequestType requestType,
            Supplier<CompletableFuture<T>> supplier, Runnable onSuccess, Consumer<Throwable> onFailure)
    {
        Stopwatch stopwatch = Stopwatch.createStarted();
        try {
            tryRequestStarted(monitoring, requestType);
            CompletableFuture<T> result = supplier.get();
            result = result.whenComplete((response, error) -> {
                try {
                    tryRequestFinished(monitoring, requestType, Duration.ofNanos(stopwatch.elapsed(TimeUnit.NANOSECONDS)),
                            error == null);
                    if (error != null) {
                        onFailure.accept(error);
                    } else {
                        onSuccess.run();
                    }
                } catch (Exception e) {
                    logger.error("YP client monitoring failure", e);
                }
            });
            return result;
        } catch (Exception e) {
            try {
                tryRequestFinished(monitoring, requestType, Duration.ofNanos(stopwatch.elapsed(TimeUnit.NANOSECONDS)), false);
                onFailure.accept(e);
            } catch (Exception ex) {
                logger.error("YP client monitoring failure", ex);
            }
            throw e;
        }
    }

    private static void tryRequestStarted(YpDiscoveryClientMonitoring monitoring, YpDiscoveryRequestType requestType) {
        try {
            monitoring.onRequestStarted(requestType);
        } catch (Exception e) {
            logger.error("Failure in onRequestStarted handler", e);
        }
    }

    private static void tryRequestFinished(YpDiscoveryClientMonitoring monitoring, YpDiscoveryRequestType requestType,
            Duration duration, boolean success)
    {
        try {
            monitoring.onRequestFinished(requestType, duration, success);
        } catch (Exception e) {
            logger.error("Failure in onRequestFinished handler", e);
        }
    }

}
