package ru.yandex.yp.discovery.impl;

import java.util.concurrent.TimeUnit;

import NYP.NServiceDiscovery.NApi.TServiceDiscoveryServiceGrpc;
import io.grpc.ConnectivityState;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;

import ru.yandex.yp.discovery.YpDiscoveryClient;
import ru.yandex.yp.discovery.YpDiscoveryClientBuilder;
import ru.yandex.yp.discovery.YpDiscoveryService;

/**
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YpDiscoveryClientImpl implements YpDiscoveryClient {

    private final ManagedChannel channel;
    private final YpDiscoveryServiceImpl discoveryService;

    public YpDiscoveryClientImpl(YpDiscoveryClientBuilder builder) {
        ManagedChannelBuilder<?> clientBuilder = ManagedChannelBuilder.forAddress(builder.getInstance().getHost(),
                builder.getInstance().getPort());
        clientBuilder
                .userAgent(builder.getUserAgent())
                .defaultLoadBalancingPolicy("round_robin");
        if (builder.usePlaintext()) {
            clientBuilder.usePlaintext();
        }
        if (!builder.getInterceptors().isEmpty()) {
            clientBuilder.intercept(builder.getInterceptors());
        }
        if (builder.getIdleTimeout().isPresent()) {
            clientBuilder.idleTimeout(builder.getIdleTimeout().get().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (builder.getKeepAliveTime().isPresent()) {
            clientBuilder.keepAliveTime(builder.getKeepAliveTime().get().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (builder.getKeepAliveTimeout().isPresent()) {
            clientBuilder.keepAliveTimeout(builder.getKeepAliveTimeout().get().toNanos(), TimeUnit.NANOSECONDS);
        }
        if (builder.getKeepAliveWithoutCalls().isPresent()) {
            clientBuilder.keepAliveWithoutCalls(builder.getKeepAliveWithoutCalls().get());
        }
        if (builder.getMaxInboundMessageSize().isPresent()) {
            clientBuilder.maxInboundMessageSize(builder.getMaxInboundMessageSize().get());
        }
        if (builder.getMaxInboundMetadataSize().isPresent()) {
            clientBuilder.maxInboundMetadataSize(builder.getMaxInboundMetadataSize().get());
        }
        this.channel = clientBuilder.build();
        TServiceDiscoveryServiceGrpc.TServiceDiscoveryServiceFutureStub discoveryServiceStub = TServiceDiscoveryServiceGrpc
                .newFutureStub(channel)
                .withInterceptors(new YpRequestIdInterceptor());
        if (builder.getCredentialsProvider().isPresent()) {
            discoveryServiceStub = discoveryServiceStub
                    .withCallCredentials(new YpCallCredentials(builder.getCredentialsProvider().get()));
        }
        this.discoveryService = new YpDiscoveryServiceImpl(discoveryServiceStub, builder.getMonitoring(), builder);
    }

    @Override
    public ConnectivityState getChannelState(boolean requestConnection) {
        return channel.getState(requestConnection);
    }

    @Override
    public boolean shutdown(long timeout, TimeUnit timeUnit) {
        return shutdown(channel, timeout, timeUnit, false);
    }

    @Override
    public boolean shutdownNow(long timeout, TimeUnit timeUnit) {
        return shutdown(channel, timeout, timeUnit, true);
    }

    @Override
    public YpDiscoveryService serviceDiscoveryService() {
        return discoveryService;
    }

    private boolean shutdown(ManagedChannel channel, long timeout, TimeUnit timeUnit, boolean now) {
        if (now) {
            channel.shutdownNow();
        } else {
            channel.shutdown();
        }
        try {
            return channel.awaitTermination(timeout, timeUnit);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        return false;
    }

}
