package ru.yandex.yp.discovery.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import NYP.NServiceDiscovery.NApi.Api;
import NYP.NServiceDiscovery.NApi.TServiceDiscoveryServiceGrpc;
import com.google.common.util.concurrent.ListenableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.yp.discovery.YpDiscoveryClientBuilder;
import ru.yandex.yp.discovery.YpDiscoveryClientMonitoring;
import ru.yandex.yp.discovery.YpDiscoveryRequestType;
import ru.yandex.yp.discovery.YpDiscoveryService;
import ru.yandex.yp.discovery.model.EndpointsResolveRequest;
import ru.yandex.yp.discovery.model.EndpointsResolveResponse;
import ru.yandex.yp.discovery.model.YpEndpoint;
import ru.yandex.yp.discovery.model.YpEndpointResolveStatus;
import ru.yandex.yp.discovery.model.YpEndpointSet;

/**
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public class YpDiscoveryServiceImpl implements YpDiscoveryService {

    private static final Logger logger = LoggerFactory.getLogger(YpDiscoveryServiceImpl.class);

    private final Supplier<TServiceDiscoveryServiceGrpc.TServiceDiscoveryServiceFutureStub> discoveryServiceSupplier;
    private final YpDiscoveryClientMonitoring monitoring;
    private final String defaultClientName;

    public YpDiscoveryServiceImpl(TServiceDiscoveryServiceGrpc.TServiceDiscoveryServiceFutureStub discoveryService,
            YpDiscoveryClientMonitoring monitoring, YpDiscoveryClientBuilder clientBuilder)
    {
        if (clientBuilder.getCallTimeout().isPresent()) {
            this.discoveryServiceSupplier = () -> discoveryService
                    .withDeadlineAfter(clientBuilder.getCallTimeout().get().toNanos(), TimeUnit.NANOSECONDS);
        } else {
            this.discoveryServiceSupplier = () -> discoveryService;
        }
        this.monitoring = monitoring;
        this.defaultClientName = clientBuilder.getClientName();
    }

    @Override
    public CompletableFuture<EndpointsResolveResponse> resolveEndpoints(EndpointsResolveRequest request) {
        logger.debug("Calling resolveEndpoints({})...", request);
        return YpClientMonitoringUtils.measure(monitoring, YpDiscoveryRequestType.RESOLVE_ENDPOINTS, () -> {
            Api.TReqResolveEndpoints.Builder requestBuilder = Api.TReqResolveEndpoints.newBuilder();
            requestBuilder.setClusterName(request.getClusterName());
            requestBuilder.setEndpointSetId(request.getEndpointSetId());
            requestBuilder.setClientName(request.getClientName().orElse(defaultClientName));
            if (!request.getLabelSelectors().isEmpty()) {
                requestBuilder.addAllLabelSelectors(request.getLabelSelectors());
            }
            if (request.getWatchToken().isPresent()) {
                requestBuilder.setWatchToken(request.getWatchToken().get());
            }
            if (request.getRuid().isPresent()) {
                requestBuilder.setRuid(request.getRuid().get());
            }
            ListenableFuture<Api.TRspResolveEndpoints> result = discoveryServiceSupplier.get()
                    .resolveEndpoints(requestBuilder.build());
            return new YpCompletableFuture<>(result).thenApply(this::mapResponse);
        }, () -> logger.debug("Successfully called resolveEndpoints({})", request),
                e -> logger.debug("Failed to call resolveEndpoints(" + request + ")", e));
    }

    private EndpointsResolveResponse mapResponse(Api.TRspResolveEndpoints response) {
        YpEndpointResolveStatus status = YpEndpointResolveStatus.from(response.getResolveStatus());
        YpEndpointSet endpointSet = null;
        if (response.hasEndpointSet()) {
            List<YpEndpoint> endpoints = response.getEndpointSet().getEndpointsList().stream()
                    .map(e -> new YpEndpoint(e.getId(), e.getProtocol(), e.getFqdn(), e.getIp4Address(),
                            e.getIp6Address(), e.getPort(), new ArrayList<>(e.getLabelSelectorResultsList()),
                            e.getReady()))
                    .collect(Collectors.toList());
            endpointSet = new YpEndpointSet(response.getEndpointSet().getEndpointSetId(), endpoints);
        }
        return new EndpointsResolveResponse(response.getTimestamp(), endpointSet, status, response.getWatchToken(),
                response.getHost(), response.getRuid());
    }

}
