package ru.yandex.yp.discovery.model;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * YP endpoints resolve request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class EndpointsResolveRequest {

    private final String clusterName;
    private final String endpointSetId;
    private final String clientName;
    private final List<String> labelSelectors;
    private final String watchToken;
    private final String ruid;

    private EndpointsResolveRequest(String clusterName, String endpointSetId, String clientName,
            List<String> labelSelectors, String watchToken, String ruid)
    {
        this.clusterName = clusterName;
        this.endpointSetId = endpointSetId;
        this.clientName = clientName;
        this.labelSelectors = labelSelectors;
        this.watchToken = watchToken;
        this.ruid = ruid;
    }

    /**
     * Create resolve request builder.
     *
     * @param clusterName YP cluster name.
     * @param endpointSetId YP endpoint set id.
     * @return {@link Builder}.
     */
    public static Builder builder(String clusterName, String endpointSetId) {
        return new Builder(clusterName, endpointSetId);
    }

    /**
     * Return YP cluster name.
     *
     * @return YP cluster name.
     */
    public String getClusterName() {
        return clusterName;
    }

    /**
     * Return YP endpoint set id.
     *
     * @return YP endpoint set id.
     */
    public String getEndpointSetId() {
        return endpointSetId;
    }

    /**
     * Return client name.
     *
     * @return Client name.
     */
    public Optional<String> getClientName() {
        return Optional.ofNullable(clientName);
    }

    /**
     * Return paths list in /labels, to request values for.
     *
     * @return Paths list in /labels, to request values for.
     */
    public List<String> getLabelSelectors() {
        return labelSelectors;
    }

    /**
     * Return watch token, deprecated.
     *
     * @return Watch token, deprecated.
     */
    @Deprecated
    public Optional<String> getWatchToken() {
        return Optional.ofNullable(watchToken);
    }

    /**
     * Return request unique id.
     * @return Request unique id.
     */
    public Optional<String> getRuid() {
        return Optional.ofNullable(ruid);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        EndpointsResolveRequest that = (EndpointsResolveRequest) o;
        return Objects.equals(clusterName, that.clusterName)
                && Objects.equals(endpointSetId, that.endpointSetId)
                && Objects.equals(clientName, that.clientName)
                && Objects.equals(labelSelectors, that.labelSelectors)
                && Objects.equals(watchToken, that.watchToken)
                && Objects.equals(ruid, that.ruid);
    }

    @Override
    public int hashCode() {
        return Objects.hash(clusterName, endpointSetId, clientName, labelSelectors, watchToken, ruid);
    }

    @Override
    public String toString() {
        return "EndpointsResolveRequest{"
                + "clusterName='" + clusterName + '\''
                + ", endpointSetId='" + endpointSetId + '\''
                + ", clientName='" + clientName + '\''
                + ", labelSelectors=" + labelSelectors
                + ", watchToken='" + watchToken + '\''
                + ", ruid='" + ruid + '\''
                + '}';
    }

    public static final class Builder {

        private final String clusterName;
        private final String endpointSetId;
        private final List<String> labelSelectors = new ArrayList<>();

        private String clientName;
        private String watchToken;
        private String ruid;

        private Builder(String clusterName, String endpointSetId) {
            this.clusterName = clusterName;
            this.endpointSetId = endpointSetId;
        }

        /**
         * Set client name.
         *
         * @param clientName Client name.
         * @return {@link Builder}.
         */
        public Builder setClientName(String clientName) {
            this.clientName = clientName;
            return this;
        }

        /**
         * Add path in /labels, to request value for.
         *
         * @param labelSelector Path in /labels, to request value for.
         * @return {@link Builder}.
         */
        public Builder addLabelSelector(String labelSelector) {
            this.labelSelectors.add(labelSelector);
            return this;
        }

        /**
         * Set watch token, deprecated.
         *
         * @param watchToken Watch token, deprecated.
         * @return {@link Builder}.
         */
        @Deprecated
        public Builder setWatchToken(String watchToken) {
            this.watchToken = watchToken;
            return this;
        }

        /**
         * Set request unique id.
         *
         * @param ruid Request unique id.
         * @return {@link Builder}.
         */
        public Builder setRuid(String ruid) {
            this.ruid = ruid;
            return this;
        }

        /**
         * Build {@link EndpointsResolveRequest}.
         *
         * @return {@link EndpointsResolveRequest}.
         */
        public EndpointsResolveRequest build() {
            return new EndpointsResolveRequest(clusterName, endpointSetId, clientName, labelSelectors, watchToken, ruid);
        }

    }

}
