package ru.yandex.yp.discovery.model;

import java.util.Objects;
import java.util.Optional;

/**
 * YP endpoints resolve response.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class EndpointsResolveResponse {

    private final long timestamp;
    private final YpEndpointSet endpointSet;
    private final YpEndpointResolveStatus resolveStatus;
    private final String watchToken;
    private final String host;
    private final String ruid;

    /**
     * Create YP endpoints resolve response.
     *
     * @param timestamp YP timestamp for returned endpoints.
     * @param endpointSet YP endpoint set.
     * @param resolveStatus Resolution status.
     * @param watchToken Watch token, deprecated.
     * @param host YP host which returned the response.
     * @param ruid Request unique id, same as request, random otherwise.
     */
    public EndpointsResolveResponse(long timestamp, YpEndpointSet endpointSet,
            YpEndpointResolveStatus resolveStatus, @Deprecated String watchToken, String host, String ruid)
    {
        this.timestamp = timestamp;
        this.endpointSet = endpointSet;
        this.resolveStatus = resolveStatus;
        this.watchToken = watchToken;
        this.host = host;
        this.ruid = ruid;
    }

    /**
     * Return YP timestamp for returned endpoints.
     *
     * @return YP timestamp for returned endpoints.
     */
    public long getTimestamp() {
        return timestamp;
    }

    /**
     * Return YP endpoint set.
     *
     * @return YP endpoint set.
     */
    public Optional<YpEndpointSet> getEndpointSet() {
        return Optional.ofNullable(endpointSet);
    }

    /**
     * Return resolution status.
     *
     * @return Resolution status.
     */
    public YpEndpointResolveStatus getResolveStatus() {
        return resolveStatus;
    }

    /**
     * Return watch token, deprecated.
     * @return Watch token, deprecated.
     */
    @Deprecated
    public String getWatchToken() {
        return watchToken;
    }

    /**
     * Return YP host which returned the response.
     * @return YP host which returned the response.
     */
    public String getHost() {
        return host;
    }

    /**
     * Return request unique id, same as request, random otherwise.
     * @return Request unique id, same as request, random otherwise.
     */
    public String getRuid() {
        return ruid;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        EndpointsResolveResponse that = (EndpointsResolveResponse) o;
        return timestamp == that.timestamp
                && resolveStatus == that.resolveStatus
                && Objects.equals(endpointSet, that.endpointSet)
                && Objects.equals(watchToken, that.watchToken)
                && Objects.equals(host, that.host)
                && Objects.equals(ruid, that.ruid);
    }

    @Override
    public int hashCode() {
        return Objects.hash(timestamp, endpointSet, resolveStatus, watchToken, host, ruid);
    }

    @Override
    public String toString() {
        return "EndpointsResolveResponse{"
                + "timestamp=" + timestamp
                + ", endpointSet=" + endpointSet
                + ", resolveStatus=" + resolveStatus
                + ", watchToken='" + watchToken + '\''
                + ", host='" + host + '\''
                + ", ruid='" + ruid + '\''
                + '}';
    }

}
