package ru.yandex.yp.discovery.model;

import java.util.List;
import java.util.Objects;

/**
 * YP endpoint.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YpEndpoint {

    private final String id;
    private final String protocol;
    private final String fqdn;
    private final String ip4Address;
    private final String ip6Address;
    private final int port;
    private final List<String> labelSelectorResults;
    private final boolean ready;

    /**
     * Create YP endpoint.
     *
     * @param id YP endpoint id.
     * @param protocol YP endpoint protocol.
     * @param fqdn YP endpoint FQDN.
     * @param ip4Address YP endpoint IPv4 address.
     * @param ip6Address YP endpoint IPv6 address.
     * @param port YP endpoint port.
     * @param labelSelectorResults Label selector values, as specified in request.
     */
    public YpEndpoint(String id, String protocol, String fqdn, String ip4Address, String ip6Address,
            int port, List<String> labelSelectorResults, boolean ready)
    {
        this.id = id;
        this.protocol = protocol;
        this.fqdn = fqdn;
        this.ip4Address = ip4Address;
        this.ip6Address = ip6Address;
        this.port = port;
        this.labelSelectorResults = labelSelectorResults;
        this.ready = ready;
    }

    /**
     * Return YP endpoint id.
     *
     * @return YP endpoint id.
     */
    public String getId() {
        return id;
    }

    /**
     * Return YP endpoint protocol.
     *
     * @return YP endpoint protocol.
     */
    public String getProtocol() {
        return protocol;
    }

    /**
     * Return YP endpoint FQDN.
     *
     * @return YP endpoint FQDN.
     */
    public String getFqdn() {
        return fqdn;
    }

    /**
     * Return YP endpoint IPv4 address.
     *
     * @return YP endpoint IPv4 address.
     */
    public String getIp4Address() {
        return ip4Address;
    }

    /**
     * Return YP endpoint IPv6 address.
     *
     * @return YP endpoint IPv6 address.
     */
    public String getIp6Address() {
        return ip6Address;
    }

    /**
     * Return YP endpoint port.
     *
     * @return YP endpoint port.
     */
    public int getPort() {
        return port;
    }

    /**
     * Return Label selector values, as specified in request.
     * @return Label selector values, as specified in request.
     */
    public List<String> getLabelSelectorResults() {
        return labelSelectorResults;
    }

    /**
     * Return instance readiness flag
     * @return instance readiness flag
     */
    public boolean isReady() {
        return ready;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YpEndpoint that = (YpEndpoint) o;
        return port == that.port
                && Objects.equals(id, that.id)
                && Objects.equals(protocol, that.protocol)
                && Objects.equals(fqdn, that.fqdn)
                && Objects.equals(ip4Address, that.ip4Address)
                && Objects.equals(ip6Address, that.ip6Address)
                && Objects.equals(labelSelectorResults, that.labelSelectorResults)
                && Objects.equals(ready, that.ready);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, protocol, fqdn, ip4Address, ip6Address, port, labelSelectorResults, ready);
    }

    @Override
    public String toString() {
        return "YpEndpoint{"
                + "id='" + id + '\''
                + ", protocol='" + protocol + '\''
                + ", fqdn='" + fqdn + '\''
                + ", ip4Address='" + ip4Address + '\''
                + ", ip6Address='" + ip6Address + '\''
                + ", port=" + port
                + ", labelSelectorResults=" + labelSelectorResults
                + ", ready=" + ready
                + '}';
    }

}
