package ru.yandex.yp.discovery.model;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.protobuf.ByteString;

import ru.yandex.yp.discovery.YpErrorAttributeDeserializer;

/**
 * YP error metadata.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class YpError {

    private final int code;
    private final String message;
    private final Map<String, ByteString> attributes;
    private final List<YpError> innerErrors;

    /**
     * Create YP error metadata.
     *
     * @param code Error code.
     * @param message Error message.
     * @param attributes Error attributes.
     * @param innerErrors Inner errors.
     */
    public YpError(int code, String message, Map<String, ByteString> attributes, List<YpError> innerErrors) {
        this.code = code;
        this.message = message;
        this.attributes = attributes;
        this.innerErrors = innerErrors;
    }

    /**
     * Return error code.
     *
     * @return Error code.
     */
    public int getCode() {
        return code;
    }

    /**
     * Return error message.
     *
     * @return Error message.
     */
    public Optional<String> getMessage() {
        return Optional.ofNullable(message);
    }

    /**
     * Return error attributes.
     *
     * @param deserializer Error attributes deserializer.
     * @param <E> Error attributes type after deserialization.
     * @return Error attributes.
     */
    public <E> Map<String, E> getAttributes(YpErrorAttributeDeserializer<E> deserializer) {
        return attributes.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey,
                e -> deserializer.deserialize(e.getKey(), e.getValue())));
    }

    /**
     * Return inner errors.
     *
     * @return Inner errors.
     */
    public List<YpError> getInnerErrors() {
        return innerErrors;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        YpError ypError = (YpError) o;
        return code == ypError.code
                && Objects.equals(message, ypError.message)
                && Objects.equals(attributes, ypError.attributes)
                && Objects.equals(innerErrors, ypError.innerErrors);
    }

    @Override
    public int hashCode() {
        return Objects.hash(code, message, attributes, innerErrors);
    }

    @Override
    public String toString() {
        return "YpError{"
                + "code=" + code
                + ", message=" + message
                + ", attributes=" + attributes
                + ", innerErrors=" + innerErrors
                + '}';
    }

}
