#include "router_api.h"

#include <infra/libs/service_iface/str_iface.h>
#include <infra/libs/version_handle/version_handle.h>

namespace NYP::NServiceDiscovery {
    using TRequestRouterPtr = TSimpleSharedPtr<NInfra::TRequestRouter<IApi>>;

    TRequestRouterPtr AddAdminRoutes(TSimpleSharedPtr<NInfra::TRequestRouter<IApi>> router) {
        (*router)
            .Add<NInfra::TEmptyRequest<NApi::TReqShutdown>, NInfra::TEmptyReply<NApi::TRspShutdown>>("/shutdown", "Shutdown", &IApi::Shutdown)
            .Add<NInfra::TEmptyRequest<NApi::TReqReopenLog>, NInfra::TProto2JsonReply<NApi::TRspReopenLog>>("/reopen_log", "Reopen log", &IApi::ReopenLog)
            .Add<NInfra::TEmptyRequest<NApi::TReqSensors>, NInfra::TRawDataReply<NApi::TRspSensors>>("/sensors", "Show sensors", &IApi::Sensors)
            .Add<NInfra::TEmptyRequest<NApi::TReqSensors>, NInfra::TRawDataReply<NApi::TRspSensors>>("/dynamic_sensors", "Show dynamic sensors", &IApi::DynamicSensors)
            .Add<NInfra::TEmptyRequest<NApi::TReqConfig>, NInfra::TRawDataReply<NApi::TRspConfig>>("/config", "Show config", &IApi::Config);

        return router;
    }

    TRequestRouterPtr AddDiscoveryRoutes(TSimpleSharedPtr<NInfra::TRequestRouter<IApi>> router) {
        (*router)
            .Add<NInfra::TEmptyRequest<NApi::TReqPing>, NInfra::TProto2JsonReply<NApi::TRspPing>>("/ping", "Ping", &IApi::Ping)
            .Add<NInfra::TSerializedProtoRequest<NApi::TReqResolveEndpoints>, NInfra::TSerializedProtoReply<NApi::TRspResolveEndpoints>>("/resolve_endpoints", "Resolve endpoints", &IApi::ResolveEndpoints)
            .Add<NInfra::TJson2ProtoRequest<NApi::TReqResolveEndpoints>, NInfra::TProto2JsonReply<NApi::TRspResolveEndpoints>>("/resolve_endpoints/json", "Resolve endpoints", &IApi::ResolveEndpoints)
            .Add<NInfra::TSerializedProtoRequest<NApi::TReqResolveNode>, NInfra::TSerializedProtoReply<NApi::TRspResolveNode>>("/resolve_node", "Resolve node", &IApi::ResolveNode)
            .Add<NInfra::TJson2ProtoRequest<NApi::TReqResolveNode>, NInfra::TProto2JsonReply<NApi::TRspResolveNode>>("/resolve_node/json", "Resolve node", &IApi::ResolveNode)
            .Add<NInfra::TSerializedProtoRequest<NApi::TReqResolvePods>, NInfra::TSerializedProtoReply<NApi::TRspResolvePods>>("/resolve_pods", "Resolve pods", &IApi::ResolvePods)
            .Add<NInfra::TJson2ProtoRequest<NApi::TReqResolvePods>, NInfra::TProto2JsonReply<NApi::TRspResolvePods, false, NProtobufJson::TProto2JsonConfig::EnumNumber, true>>("/resolve_pods/json", "Resolve pods", &IApi::ResolvePods);

        return router;
    }

    TRequestRouterPtr AddYpReplicaRoutes(TSimpleSharedPtr<NInfra::TRequestRouter<IApi>> router, const TConfig& config) {
        for (const auto& cluster : config.GetYPEndpointClusterConfigs()) {
            (*router)
                .Add<NInfra::TEmptyRequest<NApi::TReqListBackups>, NInfra::TProto2JsonReply<NApi::TRspListBackups>>(
                    TString::Join("/backup/", cluster.GetName(), "/endpoint/list/json"),
                    TString::Join("List backups for ", cluster.GetName(), ", endpoints instance (json)"),
                    &IApi::ListEndpointBackups
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqListBackups>, NInfra::TSerializedProtoReply<NApi::TRspListBackups>>(
                    TString::Join("/backup/", cluster.GetName(), "/endpoint/list"),
                    TString::Join("List backups for ", cluster.GetName(), ", endpoints instance"),
                    &IApi::ListEndpointBackups
                )
                .Add<NInfra::TSerializedProtoRequest<NApi::TReqRollbackToBackup>, NInfra::TProto2JsonReply<NApi::TRspRollbackToBackup>>(
                    TString::Join("/rollback/", cluster.GetName(), "/endpoint/backup"),
                    TString::Join("Rollback to backup for ", cluster.GetName(), ", endpoint instance (json)"),
                    &IApi::RollbackToEndpointBackup
                )
                .Add<NInfra::TJson2ProtoRequest<NApi::TReqRollbackToBackup>, NInfra::TSerializedProtoReply<NApi::TRspRollbackToBackup>>(
                    TString::Join("/rollback/", cluster.GetName(), "/endpoint/backup"),
                    TString::Join("Rollback to backup for ", cluster.GetName(), ", endpoint instance"),
                    &IApi::RollbackToEndpointBackup
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqStartUpdates>, NInfra::TRawDataReply<NApi::TRspStartUpdates>>(
                    TString::Join("/updates/", cluster.GetName(), "/endpoint/start"),
                    "Start endpoint updates",
                    &IApi::StartEndpointUpdates
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqStopUpdates>, NInfra::TRawDataReply<NApi::TRspStopUpdates>>(
                    TString::Join("/updates/", cluster.GetName(), "/endpoint/stop"),
                    "Stop endpoint updates",
                    &IApi::StopEndpointUpdates
                );
        }
        for (const auto& cluster : config.GetYPPodClusterConfigs()) {
            (*router)
                .Add<NInfra::TEmptyRequest<NApi::TReqListBackups>, NInfra::TProto2JsonReply<NApi::TRspListBackups>>(
                    TString::Join("/backup/", cluster.GetName(), "/pod/list/json"),
                    TString::Join("List backups for ", cluster.GetName(), ", pods instance (json)"),
                    &IApi::ListPodBackups
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqListBackups>, NInfra::TSerializedProtoReply<NApi::TRspListBackups>>(
                    TString::Join("/backup/", cluster.GetName(), "/pod/list"),
                    TString::Join("List backups for ", cluster.GetName(), ", pods instance"),
                    &IApi::ListPodBackups
                )
                .Add<NInfra::TSerializedProtoRequest<NApi::TReqRollbackToBackup>, NInfra::TProto2JsonReply<NApi::TRspRollbackToBackup>>(
                    TString::Join("/rollback/", cluster.GetName(), "/pod/backup"),
                    TString::Join("Rollback to backup for ", cluster.GetName(), ", pod instance (json)"),
                    &IApi::RollbackToPodBackup
                )
                .Add<NInfra::TJson2ProtoRequest<NApi::TReqRollbackToBackup>, NInfra::TSerializedProtoReply<NApi::TRspRollbackToBackup>>(
                    TString::Join("/rollback/", cluster.GetName(), "/pod/backup"),
                    TString::Join("Rollback to backup for ", cluster.GetName(), ", pod instance"),
                    &IApi::RollbackToPodBackup
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqStartUpdates>, NInfra::TRawDataReply<NApi::TRspStartUpdates>>(
                    TString::Join("/updates/", cluster.GetName(), "/pod/start"),
                    "Start pod updates",
                    &IApi::StartPodUpdates
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqStopUpdates>, NInfra::TRawDataReply<NApi::TRspStopUpdates>>(
                    TString::Join("/updates/", cluster.GetName(), "/pod/stop"),
                    "Stop pod updates",
                    &IApi::StopPodUpdates
                );
        }
        for (const auto& cluster : config.GetYPNodeClusterConfigs()) {
            (*router)
                .Add<NInfra::TEmptyRequest<NApi::TReqListBackups>, NInfra::TProto2JsonReply<NApi::TRspListBackups>>(
                    TString::Join("/backup/", cluster.GetName(), "/node/list/json"),
                    TString::Join("List backups for ", cluster.GetName(), ", node instance (json)"),
                    &IApi::ListNodeBackups
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqListBackups>, NInfra::TSerializedProtoReply<NApi::TRspListBackups>>(
                    TString::Join("/backup/", cluster.GetName(), "/node/list"),
                    TString::Join("List backups for ", cluster.GetName(), ", node instance"),
                    &IApi::ListNodeBackups
                )
                .Add<NInfra::TSerializedProtoRequest<NApi::TReqRollbackToBackup>, NInfra::TProto2JsonReply<NApi::TRspRollbackToBackup>>(
                    TString::Join("/rollback/", cluster.GetName(), "/node/backup"),
                    TString::Join("Rollback to backup for ", cluster.GetName(), ", node instance (json)"),
                    &IApi::RollbackToNodeBackup
                )
                .Add<NInfra::TJson2ProtoRequest<NApi::TReqRollbackToBackup>, NInfra::TSerializedProtoReply<NApi::TRspRollbackToBackup>>(
                    TString::Join("/rollback/", cluster.GetName(), "/node/backup"),
                    TString::Join("Rollback to backup for ", cluster.GetName(), ", node instance"),
                    &IApi::RollbackToNodeBackup
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqStartUpdates>, NInfra::TRawDataReply<NApi::TRspStartUpdates>>(
                    TString::Join("/updates/", cluster.GetName(), "/node/start"),
                    "Start node updates",
                    &IApi::StartNodeUpdates
                )
                .Add<NInfra::TEmptyRequest<NApi::TReqStopUpdates>, NInfra::TRawDataReply<NApi::TRspStopUpdates>>(
                    TString::Join("/updates/", cluster.GetName(), "/node/stop"),
                    "Stop node updates",
                    &IApi::StopNodeUpdates
                );
        }
        return router;
    }

    NInfra::TRequestRouterPtr CreateAdminRouter(IApi& service, const TConfig& config) {
        auto router = MakeSimpleShared<NInfra::TRequestRouter<IApi>>(service);
        router = AddAdminRoutes(router);
        router = AddYpReplicaRoutes(router, config);
        NInfra::NVersionHandle::AddBinaryHandlers(*router);
        return router;
    }

    NInfra::TRequestRouterPtr CreateDiscoveryRouter(IApi& service) {
        return AddDiscoveryRoutes(MakeSimpleShared<NInfra::TRequestRouter<IApi>>(service));;
    }
}
